/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.NameHelper;

/**
 * DAO for manipulating newsletter categories.
 *
 */
public class CategoryDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role */
    public static final String ROLE = CategoryDAO.class.getName();
    
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** CategoryProvider */
    private CategoryProviderExtensionPoint _categoryProviderEP;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);

    }
    
    /**
     * Creates a newsletter category.
     * @param parentId The id of the parent of the future category
     * @param title The title
     * @param description The description
     * @param siteName The site name
     * @param lang The language
     * @return The id and parentId of the created category, or an error
     */
    @Callable(rights = "Plugins_Newsletter_Right_HandleCategory")
    public Map<String, String> createCategory (String parentId, String title, String description, String siteName, String lang)
    {
        Map<String, String> result = new HashMap<>();
        
        String name = NameHelper.filterName(title);
        
        String realParentId = parentId;
        
        try
        {
            if (realParentId.startsWith("provider_"))
            {
                CategoryProvider provider = _categoryProviderEP.getExtension(realParentId.substring("provider_".length()));
                realParentId = provider.getRootId(siteName, lang);
            }
            
            DefaultTraversableAmetysObject<?> parent = _resolver.resolveById(realParentId);
            
            JCRCategory subCategory = parent.createChild(name, "ametys:category");
            subCategory.setTitle(title);
            subCategory.setDescription(description);
            
            if (parent instanceof JCRCategory)
            {
                JCRCategory parentCategory = (JCRCategory) parent;
                String templateName = parentCategory.getTemplate();
                if (templateName != null)
                {
                    subCategory.setTemplate(templateName);
                }
            }
            
            parent.saveChanges();
            
            result.put("id", subCategory.getId());
            result.put("parentId", parentId);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().error("Unable to add a new category : the tag category '" + realParentId + "' does not exist anymore", e);
            result.put("message", "unknown-category");
        }
        catch (RepositoryIntegrityViolationException e)
        {
            getLogger().error("Unable to add a new category : a category with name '" + name + "' already exist", e);
            result.put("message", "already-exist");
        }
       
        return result;
    }
    
    /**
     * Edits a newsletter category.
     * @param id The id of the category to edit
     * @param title The title
     * @param description The description
     * @return The id and title of the modified category, or an error
     */
    @Callable(rights = "Plugins_Newsletter_Right_HandleCategory")
    public Map<String, String> editCategory (String id, String title, String description)
    {
        Map<String, String> result = new HashMap<>();
        
        try
        {
            JCRCategory jcrCategory = _resolver.resolveById(id);
            jcrCategory.setTitle(title);
            jcrCategory.setDescription(description);
            
            jcrCategory.saveChanges();
            
            result.put("id", jcrCategory.getId());
            result.put("title", title);
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().error("Unable to update category : the category '" + id + "' does not exist anymore", e);
            result.put("message", "unknown-category");
        }
        
        return result;
    }
    
    /**
     * Deletes a newsletter category.
     * @param id The id of the category to delete
     * @return The id of the deleted category, or an error
     */
    @Callable(rights = "Plugins_Newsletter_Right_HandleCategory")
    public Map<String, String> deleteCategory (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        try
        {
            JCRCategory jcrCategory = _resolver.resolveById(id);
            ModifiableAmetysObject parent = jcrCategory.getParent();
            jcrCategory.remove();
            
            parent.saveChanges();
            
            result.put("id", jcrCategory.getId());
        }
        catch (UnknownAmetysObjectException e)
        {
            getLogger().error("Unable to delete tag category : the category '" + id + "' does not exist anymore", e);
            result.put("message", "unknown-category");
        }
        
        return result;
    }
    
    /**
     * Gets the properties of given categories.
     * @param categoryIds The ids of categories
     * @return The properties of categories in a result map
     */
    @Callable(rights = {"Plugins_Newsletter_Right_AccessNewsletter", "Newsletters_Right_Service_Newsletters"})
    public List<Map<String, Object>> getCategories (List<String> categoryIds)
    {
        List<Map<String, Object>> categories = new ArrayList<>();
        
        for (String categoryId : categoryIds)
        {
            categories.add(getCategory(categoryId));
        }
        
        return categories;
    }
    
    /**
     * Gets the properties of given categories.
     * @param categoryId The id of the category to look for
     * @return The properties of categories in a result map
     */
    @Callable(rights = {"Plugins_Newsletter_Right_AccessNewsletter"})
    public Map<String, Object> getCategory (String categoryId)
    {
        for (String providerId : _categoryProviderEP.getExtensionsIds())
        {
            CategoryProvider provider = _categoryProviderEP.getExtension(providerId);
            if (provider.hasCategory(categoryId))
            {
                Category category = provider.getCategory(categoryId);
                
                Map<String, Object> infos = new HashMap<>();
                
                infos.put("id", category.getId());
                infos.put("title", category.getTitle());
                infos.put("description", category.getDescription());
                infos.put("name", category.getName());
                infos.put("template", category.getTemplate());
                infos.put("mode", provider.isWritable() ? "write" : "read");
                infos.put("lang", category.getLang());
                infos.put("siteName", category.getSiteName());
                infos.put("parentId", category.getParentId());
                infos.put("automaticIds", provider.getAutomaticIds(categoryId));
                
                return infos;
            }
        }
        
        return null;
    }
    
    /**
     * Applies a template to a given newsletter category.
     * @param categoryIds The ids of the categories to edit.
     * @param template The template to set.
     * @return An empty map
     */
    @Callable(rights = "Plugins_Newsletter_Right_HandleCategory")
    public Map<String, Object> applyTemplates (List<String> categoryIds, String template)
    {
        Set<String> providerIDs = _categoryProviderEP.getExtensionsIds();
        
        for (String id : categoryIds)
        {
            for (String providerID : providerIDs)
            {
                CategoryProvider provider = _categoryProviderEP.getExtension(providerID);
                if (provider.hasCategory(id))
                {
                    Category category = provider.getCategory(id);
                    provider.setTemplate(category, template);
                }
                
            }
        }
        
        return new HashMap<>();
    }

}
