/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

/**
 * This class is in charge to load and initialize newsletter category providers.
 */
public class CategoryProviderExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<CategoryProvider>
{
    /** Avalon Role */
    public static final String ROLE = CategoryProviderExtensionPoint.class.getName();
    
    /**
     * Get all the categories for a given siteName and language.
     * @param siteName the site name.
     * @param language the language.
     * @return a Collection containing all the categories.
     */
    public Collection<Category> getCategories(String siteName, String language)
    {
        List<Category> categories = new ArrayList<>();
        
        for (String id : getExtensionsIds())
        {
            CategoryProvider provider = getExtension(id);
            
            categories.addAll(provider.getCategories(siteName, language));
        }
        
        return categories;
    }
    
    
    /**
     * Get a category by its ID.
     * @param categoryId the category ID.
     * @return the Category or null if not found.
     */
    public Category getCategory(String categoryId)
    {
        for (String id : getExtensionsIds())
        {
            CategoryProvider provider = getExtension(id);
            if (provider.hasCategory(categoryId))
            {
                return provider.getCategory(categoryId);
            }
        }
        
        return null;
    }
    
    /**
     * Tests if a category exists.
     * @param categoryId the category ID.
     * @return true if the category exists, false otherwise..
     */
    public boolean hasCategory(String categoryId)
    {
        for (String id : getExtensionsIds())
        {
            CategoryProvider provider = getExtension(id);
            if (provider.hasCategory(categoryId))
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Get a category's provider.
     * @param categoryId the category ID.
     * @return the CategoryProvider or null if not found.
     */
    public CategoryProvider getCategoryProvider(String categoryId)
    {
        for (String id : getExtensionsIds())
        {
            CategoryProvider provider = getExtension(id);
            if (provider.hasCategory(categoryId))
            {
                return provider;
            }
        }
        
        return null;
    }
    
}
