/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.io.InputStream;
import java.util.Date;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.runtime.i18n.I18nizableText;

/**
 * Represent a newsletter template. 
 */
public class NewsletterTemplate extends AbstractLogEnabled implements Serviceable, Component
{
    private SourceResolver _sourceResolver;
    
    private String _id;
    private String _skinId;
    private I18nizableText _label;
    private I18nizableText _description;
    private String _smallImage;
    private String _mediumImage;
    private String _largeImage;
    private long _lastConfUpdate; 
    
    /**
     * Creates a template
     * @param skinId The skin id
     * @param templateId The template id
     */
    public NewsletterTemplate(String skinId, String templateId)
    {
        this._skinId = skinId;
        this._id = templateId;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    /**
     * The template id
     * @return the id
     */
    public String getId()
    {
        return _id;
    }
    
    /**
     * The template label
     * @return The label
     */
    public I18nizableText getLabel()
    {
        return _label;
    }
    /**
     * The template description
     * @return The description. Can not be null but can be empty
     */
    public I18nizableText getDescription()
    {
        return _description;
    }
    
    /**
     * The small image file uri
     * @return The small image file uri
     */
    public String getSmallImage()
    {
        return _smallImage;
    }

    /**
     * The medium image file uri
     * @return The medium image file uri
     */
    public String getMediumImage()
    {
        return _mediumImage;
    }
    
    /**
     * The large image file uri
     * @return The large image file uri
     */
    public String getLargeImage()
    {
        return _largeImage;
    }
    
    
    /**
     * Refresh the configuration values
     */
    public void refreshValues ()
    {
        Source configurationFile = null;
        try
        {
            configurationFile = _sourceResolver.resolveURI("skin:" + _skinId + "://newsletter/" + this._id + "/template.xml");
            if (configurationFile.exists())
            {
                if (_lastConfUpdate < configurationFile.getLastModified())
                {
                    _defaultValues();

                    _lastConfUpdate = configurationFile.getLastModified();
                    try (InputStream is = configurationFile.getInputStream())
                    {
                        Configuration configuration = new DefaultConfigurationBuilder().build(is);
                        
                        this._label = _configureI18n(configuration.getChild("label", false), this._label);
                        this._description = _configureI18n(configuration.getChild("description", false), this._description);
                        this._smallImage = _configureThumbnail(configuration.getChild("thumbnail").getChild("small").getValue(null), this._smallImage);
                        this._mediumImage = _configureThumbnail(configuration.getChild("thumbnail").getChild("medium").getValue(null), this._mediumImage);
                        this._largeImage = _configureThumbnail(configuration.getChild("thumbnail").getChild("marge").getValue(null), this._largeImage);
                    }
                }
            }
            else
            {
                _defaultValues();
            }
        }
        catch (Exception e)
        {
            _defaultValues();
        }
        finally
        {
            _sourceResolver.release(configurationFile);
        }
    }
    
    private void _defaultValues()
    {
        _lastConfUpdate = new Date().getTime();
        
        this._label = new I18nizableText(this._id);
        this._description = new I18nizableText("");
        this._smallImage = "/plugins/newsletter/resources/img/category/template_16.png";
        this._mediumImage = "/plugins/newsletter/resources/img/category/template_32.png";
        this._largeImage = "/plugins/newsletter/resources/img/category/template_48.png";
    }
    
    private I18nizableText _configureI18n(Configuration child, I18nizableText defaultValue) throws ConfigurationException
    {
        if (child != null)
        {
            String value = child.getValue();
            if (child.getAttributeAsBoolean("i18n", false))
            {
                return new I18nizableText("skin." + this._skinId, value);
            }
            else
            {
                return new I18nizableText(value);
            }
        }
        else
        {
            return defaultValue;
        }
    }
    
    private String _configureThumbnail(String value, String defaultImage)
    {
        if (value == null)
        {
            return defaultImage;
        }
        else
        {
            return "/skins/" + this._skinId + "/newsletter/" + this._id + "/resources/" + value;
        }
    }

}
