/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.content;

import java.util.Arrays;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.xpath.XPathAPI;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.ametys.cms.repository.ModifiableContent;
import org.ametys.plugins.newsletter.NewsletterDAO;
import org.ametys.plugins.repository.data.extractor.ValuesExtractor;
import org.ametys.plugins.repository.data.extractor.xml.ModelLessXMLValuesExtractor;
import org.ametys.plugins.repository.data.extractor.xml.XMLValuesExtractorAdditionalDataGetter;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;

/**
 * Generates SAX events for Content, including category, automatic and sent for Newsletters.
 * TODO NEWATTRIBUTEAPI_CONTENT: do not use type implementation but the ModelAwareDataHolder#getInternalValue when this API exist
 */
public class ContentExtractor extends org.ametys.web.content.ContentExtractor
{
    /** Avalon role. */
    public static final String NEWSLETTER_CONTENT_EXTACTOR_ROLE = ContentExtractor.class.getName();
    
    private static final String __CATEGORY_METADATA_NAME = "category";
    private static final String __AUTOMATIC_METADATA_NAME = "automatic";
    private static final String __SENT_METADATA_NAME = "sent";
    
    private ModelItemTypeExtensionPoint _metadataTypesExtensionPoints;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _metadataTypesExtensionPoints = (ModelItemTypeExtensionPoint) manager.lookup(ModelItemTypeExtensionPoint.ROLE_MODEL_LESS_BASIC);
    }
    
    @Override
    public void fillContent(ModifiableContent content, Node node, XMLValuesExtractorAdditionalDataGetter additionalDataGetter) throws Exception
    {
        super.fillContent(content, node, additionalDataGetter);
        
        if (Arrays.asList(content.getTypes()).contains(NewsletterDAO.__NEWSLETTER_CONTENT_TYPE))
        {
            Element contentElement = (Element) XPathAPI.selectSingleNode(node, "content");
            fillMetadata(content, contentElement, additionalDataGetter);
        }
    }
    
    /**
     * Fills the given newsletter with the metadata from the provided {@link Element}
     * @param content The newsletter to fill
     * @param contentElement the element to read to get the metadata
     * @param additionalDataGetter The object that will retrieve potential additional data for the content's metadata
     * @throws Exception if an error occurs
     */
    protected void fillMetadata(ModifiableContent content, Element contentElement, XMLValuesExtractorAdditionalDataGetter additionalDataGetter) throws Exception
    {
        ModifiableModelLessDataHolder internalDataHolder = content.getInternalDataHolder();
        ValuesExtractor valuesExtractor = new ModelLessXMLValuesExtractor(contentElement, additionalDataGetter, _metadataTypesExtensionPoints);
        
        Object category = valuesExtractor.extractValue(__CATEGORY_METADATA_NAME);
        internalDataHolder.setValue(__CATEGORY_METADATA_NAME, category);
        
        Object automatic = valuesExtractor.extractValue(__AUTOMATIC_METADATA_NAME);
        internalDataHolder.setValue(__AUTOMATIC_METADATA_NAME, automatic);
        
        Object sent = valuesExtractor.extractValue(__SENT_METADATA_NAME);
        internalDataHolder.setValue(__SENT_METADATA_NAME, sent);
    }
}
