/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.subscribe;

import java.util.Date;
import java.util.Set;
import java.util.UUID;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.captcha.CaptchaHelper;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.plugins.newsletter.daos.SubscribersDAO;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.site.SiteConfigurationExtensionPoint;

/**
 * Abstract action for subscription
 *
 */
public abstract class AbstractSubscribeAction extends ServiceableAction implements ThreadSafe
{
    /** The subscribers DAO */
    protected SubscribersDAO _subscribersDao;
    /** The category providers manager */
    protected CategoryProviderExtensionPoint _categoryProviderEP;
    /** The site configuration */
    protected SiteConfigurationExtensionPoint _siteConfiguration;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** Page helper */
    protected PageHelper _pageHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _subscribersDao = (SubscribersDAO) smanager.lookup(SubscribersDAO.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _siteConfiguration = (SiteConfigurationExtensionPoint) smanager.lookup(SiteConfigurationExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _pageHelper = (PageHelper) smanager.lookup(PageHelper.ROLE);
    }
    
    /**
     * Determines if the email address is valid
     * @param email The email address
     * @return true if the email is valid
     */
    protected boolean _validEmail (String email)
    {
        return !StringUtils.isBlank(email) 
                && SendMailHelper.EMAIL_VALIDATION.matcher(email).matches();
    }

    /**
     * Determines if the captcha is valid
     * @param request The request
     * @param sitemapElement the page holding the service
     * @return true if the captcha is valid
     */
    protected boolean _validCaptcha (Request request, SitemapElement sitemapElement)
    {
        if (_pageHelper.isCaptchaRequired(sitemapElement))
        {
            String captchaKey = request.getParameter("captcha-key");
            String answer = request.getParameter("captcha");

            // Validate captcha 
            if (!CaptchaHelper.checkAndInvalidate(captchaKey, answer))
            {
                return false;
            }
        }
        return true;
    }

    /**
     * Get the category
     * @param categoryID The category id
     * @return the category
     */
    protected Category _getCategory (String categoryID)
    {
        Set<String> ids = _categoryProviderEP.getExtensionsIds();
        for (String id : ids)
        {
            CategoryProvider provider = _categoryProviderEP.getExtension(id);
            if (provider.hasCategory(categoryID))
            {
                return provider.getCategory(categoryID);
            }
        }
        
        return null;
    }
    
    /**
     * Get the category
     * @param email the subscriber email
     * @param siteName the site name
     * @param categoryID The category id
     * @return the category
     */
    protected Subscriber _createSubscritpion (String email, String siteName, String categoryID)
    {
        Subscriber subscriber = new Subscriber();
        subscriber.setEmail(email);
        subscriber.setSiteName(siteName);
        subscriber.setCategoryId(categoryID);
        subscriber.setSubscribedAt(new Date());
        
        // Generate unique token
        String token = UUID.randomUUID().toString();
        subscriber.setToken(token);
        
        return subscriber;
    }
}
