/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.subscribe;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.cocoon.servlet.multipart.PartOnDisk;
import org.apache.cocoon.servlet.multipart.RejectedPart;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BOMInputStream;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.newsletter.daos.Subscriber;
import org.ametys.plugins.newsletter.daos.SubscribersDAO;

/**
 * Import subscribers from a CSV or text file.
 */
public class ImportSubscribersAction extends ServiceableAction
{
    private static final String[] _ALLOWED_EXTENSIONS = new String[] {"txt", "csv"};
    
    private static final Pattern __EMAIL_VALIDATOR = SendMailHelper.EMAIL_VALIDATION;
    
    /** The subscribers DAO. */
    protected SubscribersDAO _subscribersDao;
    
    /** The category provider extension point. */
    protected CategoryProviderExtensionPoint _categoryProviderEP;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _subscribersDao = (SubscribersDAO) smanager.lookup(SubscribersDAO.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();

        Request request = ObjectModelHelper.getRequest(objectModel);

        String siteName = parameters.getParameter("siteName", request.getParameter("siteName"));
        String categoryId = request.getParameter("categoryId");
        boolean cleanTable = "true".equals(request.getParameter("cleanSubscribers"));
        
        Part part = (Part) request.get("importFile");
        if (part instanceof RejectedPart)
        {
            request.setAttribute(JSonReader.OBJECT_TO_READ, Map.of("success", false, "error", "rejected-file"));
            return EMPTY_MAP;
        }
        
        PartOnDisk uploadedFilePart = (PartOnDisk) part;
        File uploadedFile = (uploadedFilePart != null) ? uploadedFilePart.getFile() : null;
        String filename = (uploadedFilePart != null) ? uploadedFilePart.getFileName().toLowerCase() : null;
        
        if (!FilenameUtils.isExtension(filename, _ALLOWED_EXTENSIONS))
        {
            request.setAttribute(JSonReader.OBJECT_TO_READ, Map.of("success", false, "error", "invalid-extension"));
            return EMPTY_MAP;
        }
        
        // Test if the category exists.
        Category category = _categoryProviderEP.getCategory(categoryId);
        if (category == null)
        {
            request.setAttribute(JSonReader.OBJECT_TO_READ, Map.of("success", false, "error", "unknown-category"));
            return EMPTY_MAP;
        }
        
        FileInputStream fileIS =  new FileInputStream(uploadedFile);
        BOMInputStream bomIS = new BOMInputStream(fileIS);
        
        // Extract the emails from the import file.
        Collection<String> emails = getEmails(bomIS);
        
        if (!emails.isEmpty())
        {
            // Empty the category subscribers if needed.
            if (cleanTable)
            {
                _subscribersDao.empty(categoryId, siteName);
            }
            
            // Insert the emails.
            insertSubscribers(emails, categoryId, siteName, result);
        }
        
        result.put("success", true);
        result.put("categoryId", categoryId);
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    /**
     * Extract the emails from the file.
     * @param bomIS the input stream 
     * @return a collection of the emails.
     * @throws IOException if an error occurs reading the file.
     */
    protected Collection<String> getEmails(BOMInputStream bomIS) throws IOException
    {
        Set<String> emails = new LinkedHashSet<>();
        
        for (String line : IOUtils.readLines(bomIS, StandardCharsets.UTF_8))
        {
            String[] part = line.split(";");
            String email = StringUtils.trimToEmpty(part[0]);
            if (__EMAIL_VALIDATOR.matcher(email.toLowerCase()).matches() && StringUtils.isNotBlank(email))
            {
                emails.add(email);
            }
        }
        
        return emails;
    }
    
    /**
     * Insert subscribers
     * @param emails the emails to insert
     * @param categoryId the id of category
     * @param siteName the site name of category
     * @param result the result map
     */
    protected void insertSubscribers(Collection<String> emails, String categoryId, String siteName, Map<String, Object> result)
    {
        int subscribedCount = 0;
        int existingCount = 0;
        int errorCount = 0;
        
        for (String email : emails)
        {
            try
            {
                if (_subscribersDao.getSubscriber(email, siteName, categoryId) == null)
                {
                    Subscriber subscriber = new Subscriber();
                    subscriber.setEmail(email);
                    subscriber.setSiteName(siteName);
                    subscriber.setCategoryId(categoryId);
                    subscriber.setSubscribedAt(new Date());
                    
                    // Generate unique token.
                    String token = UUID.randomUUID().toString();
                    subscriber.setToken(token);
                    
                    _subscribersDao.subscribe(subscriber);
                    
                    if (getLogger().isInfoEnabled())
                    {
                        getLogger().info("The user with email '" + email + "' subscribed to the newsletter with the token " + token);
                    }
                    
                    subscribedCount++;
                }
                else
                {
                    existingCount++;
                }
            }
            catch (Exception e)
            {
                getLogger().error("Impossible to add as a subscriber the email " + email + " in category " + categoryId + " of site " + siteName, e);
                errorCount++;
            }
        }
        
        result.put("subscribedCount", Integer.toString(subscribedCount));
        result.put("existingCount", Integer.toString(existingCount));
        result.put("errorCount", Integer.toString(errorCount));
    }

}
