/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.testsending;

import java.io.InputStream;

import javax.jcr.Repository;
import javax.jcr.Session;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.data.NamedResource;
import org.ametys.cms.data.RichText;
import org.ametys.core.util.ImageResolverHelper;
import org.ametys.web.WebConstants;
import org.ametys.web.WebHelper;
import org.ametys.web.editor.LocalURIResolver;

/**
 * Resolver for local uri in newsletters
 */
public class NewsletterLocalURIResolver extends LocalURIResolver
{
    /** resolver data type for newsletter local */
    public static final String NEWSLETTER_LOCAL_DATA_TYPE = "newsletter-local";
    private Repository _repository;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _repository = (Repository) manager.lookup(Repository.class.getName());
    }
    
    @Override
    public String getType()
    {
        return NEWSLETTER_LOCAL_DATA_TYPE;
    }
    
    @Override
    public String resolve(String uri, boolean download, boolean absolute, boolean internal)
    {
        return _resolve(uri, download, absolute, internal, null);
    }
    
    @Override
    public String resolveImage(String uri, int height, int width, boolean download, boolean absolute, boolean internal)
    {
        String suffix = height != 0 || width != 0 ? "_" + height + "x" + width : null;
        return _resolve(uri, download, absolute, internal, suffix);
    }
    
    @Override
    public String resolveBoundedImage(String uri, int maxHeight, int maxWidth, boolean download, boolean absolute, boolean internal)
    {
        String suffix = maxHeight != 0 || maxWidth != 0 ? "_max" + maxHeight + "x" + maxWidth : null;
        return _resolve(uri, download, absolute, internal, suffix);
    }
    
    @Override
    public String resolveCroppedImage(String uri, int cropHeight, int cropWidth, boolean download, boolean absolute, boolean internal)
    {
        String suffix = cropHeight != 0 || cropWidth != 0 ? "_crop" + cropHeight + "x" + cropWidth : null;
        return _resolve(uri, download, absolute, internal, suffix);
    }
    
    @Override
    public String resolveImageAsBase64(String uri, int height, int width)
    {
        return _resolveImageAsBase64(uri, height, width, 0, 0, 0, 0);
    }
    
    @Override
    public String resolveBoundedImageAsBase64(String uri, int maxHeight, int maxWidth)
    {
        return _resolveImageAsBase64(uri, 0, 0, maxHeight, maxWidth, 0, 0);
    }
    
    @Override
    public String resolveCroppedImageAsBase64(String uri, int cropHeight, int cropWidth)
    {
        return _resolveImageAsBase64(uri, 0, 0, 0, 0, cropHeight, cropWidth);
    }
    
    /**
     * Resolves a link URI for rendering purposes.<br>
     * The output must be a properly encoded path, relative to the webapp context, accessible from a browser.
     * @param uri the link URI.
     * @param download true if the pointed resource is to be downloaded.
     * @param absolute true if the url must be absolute
     * @param internal true to get an internal URI.
     * @param suffix The suffix to add to the resolved path
     * @return the path to the resource.
     */
    @Override
    protected String _resolve(String uri, boolean download, boolean absolute, boolean internal, String suffix)
    {
        Session liveSession = null;
        try
        {
            liveSession = _repository.login(WebConstants.LIVE_WORKSPACE);
            
            URIInfo infos = getInfos(uri, true, liveSession);
            
            Request request = ContextHelper.getRequest(_context);
            
            String siteName = WebHelper.getSiteName(request, infos.getContent());
            
            return _resolve(infos, uri, siteName, download, absolute, internal, suffix);
        }
        catch (Exception e)
        {
            getLogger().warn("Cannot resolve link " + uri, e);
            return "";
        }
        finally
        {
            if (liveSession != null)
            {
                liveSession.logout();
            }
        }
    }
    
    /**
     * Resolve image as base 64
     * @param uri the link URI.
     * @param height the specified height. Ignored if 0.
     * @param width the specified width. Ignored if 0.
     * @param maxHeight the maximum image height. Ignored if height or width is specified.
     * @param maxWidth the maximum image width. Ignored if height or width is specified.
     * @param cropHeight the cropping height. Ignored if 0.
     * @param cropWidth the cropping width. Ignored if 0.
     * @return a base64-encoded string representing the image.
     */
    protected String _resolveImageAsBase64(String uri, int height, int width, int maxHeight, int maxWidth, int cropHeight, int cropWidth)
    {
        Session liveSession = null;
        try
        {
            liveSession = _repository.login(WebConstants.LIVE_WORKSPACE);
            URIInfo infos = getInfos(uri, true, liveSession);
            
            RichText richText = infos.getContent().getValue(infos.getAttribute());
            NamedResource attachment = richText.getAttachment(infos.getFilename());
            
            try (InputStream dataIs = attachment.getInputStream())
            {
                return ImageResolverHelper.resolveImageAsBase64(dataIs, attachment.getMimeType(), height, width, maxHeight, maxWidth, cropHeight, cropWidth);
            }
        }
        catch (Exception e)
        {
            getLogger().warn("Cannot resolve link " + uri);
            return "";
        }
        finally
        {
            if (liveSession != null)
            {
                liveSession.logout();
            }
        }
    }
    
    @Override
    public String getMimeType(String uri)
    {
        Session liveSession = null;
        try
        {
            liveSession = _repository.login(WebConstants.LIVE_WORKSPACE);
            URIInfo infos = getInfos(uri, true, liveSession);
            
            RichText richText = infos.getContent().getValue(infos.getAttribute());
            NamedResource attachment = richText.getAttachment(infos.getFilename());
            return attachment.getMimeType();
        }
        catch (Exception e)
        {
            getLogger().warn("Cannot resolve link " + uri);
            return "";
        }
        finally
        {
            if (liveSession != null)
            {
                liveSession.logout();
            }
        }
    }
}
