/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.workflow;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Date;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.Constants;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.web.analytics.WebAnalyticsProvider;
import org.ametys.web.analytics.WebAnalyticsProviderExtensionPoint;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Send a web analytics event for every newsletter e-mail sent.
 */
public class SendWebAnalyticsEventsEngine implements Runnable
{
    
    private static final Logger _LOGGER = LoggerFactory.getLogger(SendWebAnalyticsEventsEngine.class);
    
    /** The avalon context. */
    protected Context _context;
    
    /** The cocoon environment context. */
    protected org.apache.cocoon.environment.Context _environmentContext;
    
    /** The service manager. */
    protected ServiceManager _manager;
    
    /** Is the engine initialized ? */
    protected boolean _initialized;
    
    /** The category provider extension point. */
    protected CategoryProviderExtensionPoint _categoryProviderEP;
    
    /** The site Manager. */
    protected SiteManager _siteManager;
    
    /** The web analytics provider extension point */
    protected WebAnalyticsProviderExtensionPoint _webAnalyticsProviderEP;
    
    /** The i18n utils component. */
    protected I18nUtils _i18nUtils;
    
    /** The resolver */
    protected AmetysObjectResolver _resolver;
    
    private boolean _parametrized;
    
    private Site _site;
    
    private LocalDate _newsletterDate;
    private long _newsletterNumber;
    private String _newsletterTitle;
    
    private Category _category;
    
    private int _eventCount;
    
    /**
     * Initialize the engine.
     * @param manager the avalon service manager.
     * @param context the avalon context.
     * @throws ContextException if an error occurred during initialization
     * @throws ServiceException if an error occurred during initialization
     */
    public void initialize(ServiceManager manager, Context context) throws ContextException, ServiceException
    {
        _manager = manager;
        _context = context;
        _environmentContext = (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
        
        // Lookup the needed components.
        _webAnalyticsProviderEP = (WebAnalyticsProviderExtensionPoint) manager.lookup(WebAnalyticsProviderExtensionPoint.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) manager.lookup(CategoryProviderExtensionPoint.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        
        _initialized = true;
    }
    
    /**
     * Parameterize engine
     * @param siteName The site name
     * @param newsletterContent the newsletter content
     * @param category the newsletter category
     * @param eventCount the number of events
     */
    public void parametrize(String siteName, WebContent newsletterContent, Category category, int eventCount)
    {
        if (siteName != null)
        {
            _site = _siteManager.getSite(siteName);
        }
        _category = category;
        _eventCount = eventCount;
        
        _newsletterDate = newsletterContent.getValue("newsletter-date");
        _newsletterNumber = newsletterContent.getValue("newsletter-number", false, 0L);
        _newsletterTitle = newsletterContent.getValue("title", false, "");
        
        _parametrized = true;
    }
    
    private void _checkInitialization()
    {
        if (!_initialized)
        {
            String message = "The web analytics events engine has to be properly initialized before it's run.";
            _LOGGER.error(message);
            throw new IllegalStateException(message);
        }
        if (!_parametrized)
        {
            String message = "The web analytics events engine has to be parametrized before it's run.";
            _LOGGER.error(message);
            throw new IllegalStateException(message);
        }
    }
    
    public void run()
    {
        _checkInitialization();
        
        if (_LOGGER.isInfoEnabled())
        {
            _LOGGER.info("Starting to send web analytics newsletter events.");
        }
        
        sendEvents();
        
        if (_LOGGER.isInfoEnabled())
        {
            _LOGGER.info("All mails are sent at " + new Date());
        }
    }
    
    /**
     * Send web analytics events for a given web analytics user account.
     */
    protected void sendEvents()
    {
        Site site = _resolver.resolveById(_site.getId());
        if (site.getValue("newsletter-enable-tracking", false, true))
        {
            String trackingProviderId = site.getValue("tracking-provider", true, StringUtils.EMPTY);
            if (_webAnalyticsProviderEP.hasExtension(trackingProviderId))
            {
                try
                {
                    WebAnalyticsProvider provider = _webAnalyticsProviderEP.getExtension(trackingProviderId);
                    String category = _getNewsletterCategory();
                    String label = _getNewsletterLabel();
                    
                    for (int i = 0; i < _eventCount; i++)
                    {
                        // Build the URI and send the request.
                        String uri = provider.getEventImageUri(site, category, "Sending", label, 0, false);
                        if (StringUtils.isNotBlank(uri))
                        {
                            try
                            {
                                if (i > 0)
                                {
                                    Thread.sleep(1100);
                                }
                                
                                sendEvent(uri);
                            }
                            catch (IOException e)
                            {
                                _LOGGER.error("IO error sending the event.", e);
                            }
                            catch (InterruptedException e)
                            {
                                _LOGGER.error("Error while waiting for sending mails.", e);
                            }
                        }
                    }
                }
                catch (IllegalArgumentException e)
                {
                    // the extension doesn't exist
                }
            }
        }
    }
    
    /**
     * Send the event request, given the full GIF URL.
     * @param eventUrl the full event GIF URL (with parameters).
     * @throws IOException if an error occurs sending the GIF HTTP request.
     */
    protected void sendEvent(String eventUrl) throws IOException
    {
        URL url = new URL(eventUrl);
        
        // Use a custom user agent to avoid 
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestProperty("User-Agent", "Ametys/3 (compatible; MSIE 6.0)");
        connection.setRequestMethod("GET");
        connection.connect();
        
        int responseCode = connection.getResponseCode();
        
        if (responseCode != HttpURLConnection.HTTP_OK)
        {
            _LOGGER.error("Web analytics image request returned with error (code: " + responseCode + ").");
        }
        else
        {
            // Quietly consume the stream, ignoring errors.
            consumeQuietly(connection.getInputStream());
        }
    }
    
    private String _getNewsletterCategory()
    {
        String newsletterTitle = _i18nUtils.translate(_category.getTitle(), _category.getLang());
        return "Newsletters/" + newsletterTitle;
    }
    
    private String _getNewsletterLabel()
    {
        StringBuilder label = new StringBuilder();
        label.append(_newsletterTitle);
        if (_newsletterNumber > 0)
        {
            label.append("-").append(_newsletterNumber);
        }
        if (_newsletterDate != null)
        {
            label.append("-").append(_newsletterDate.format(DateTimeFormatter.ISO_LOCAL_DATE));
        }
        
        return label.toString();
    }
    
    /**
     * Consume a stream, reading all its data and ignoring errors. 
     * @param input the input stream to consume.
     */
    protected void consumeQuietly(InputStream input)
    {
        try
        {
            byte[] buffer = new byte[256];
            while (input.read(buffer) != -1)
            {
                // Ignore.
            }
        }
        catch (IOException e)
        {
            // Ignore.
        }
        finally
        {
            // Close quietly
            try 
            {
                if (input != null) 
                {
                    input.close();
                }
            } 
            catch (IOException ioe) 
            {
                // ignore
            }
        }
    }
}
