/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
/**
 * Singleton class to handle the validate action on newsletter.
 * @private
 */
 Ext.define('Ametys.plugins.newsletter.workflow.ValidateAction', {
 	singleton: true,
 	
 	/**
 	 * This function validates the content registered by the controller
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling this function
 	 */
 	act: function(controller)
 	{
 		var contents = controller.contents;
 		var actionId = controller['workflow-action-id'];
 		
 		if (contents.length > 1)
		{
 			// Too many contents
			Ametys.Msg.show({
				title: "{{i18n PLUGINS_NEWSLETTER_WORKFLOW_VALIDATE}}",
				msg: "{{i18n PLUGINS_NEWSLETTER_WORKFLOW_SELECTION_ERROR}}",
				buttons: Ext.MessageBox.INFO,
				icon: Ext.MessageBox.ERROR
			});
			
			return;
		}
 		
 		var contentId = contents[0];
 		
 		Ametys.data.ServerComm.callMethod({
			role: "org.ametys.plugins.newsletter.NewsletterDAO",
			methodName: 'isSent',
			parameters: [contentId],
			callback: {
				handler: this._isSentCb,
				scope: this,
				arguments: {
					contentId: contentId,
					actionId: actionId,
					controller: controller
				}
			},
			errorMessage: true,
			waitMessage: true
		});
 	},

    /**
     * @private
     * Callback for #act
     * @param {boolean} alreadySent true if the newsletter was already sent once. If so, the user will be assked to resent it or not.
     * @param {Object} args Arguments transmitted
     * @param {String} args.contentId The content identifier
     * @param {String} args.actionId The workflow action id
     * @param {Ametys.ribbon.element.ui.ButtonController} args.controller The button controller call the action
     */
 	_isSentCb: function (alreadySent, args)
 	{
 		if (alreadySent)
		{
			// Validate and re-sent the newsletter ?
			Ametys.Msg.confirm("{{i18n PLUGINS_NEWSLETTER_WORKFLOW_VALIDATE}}", 
					"{{i18n PLUGINS_NEWSLETTER_WORKFLOW_VALIDATE_CONFIRM}}", 
					function (answer) {
							if (answer != 'cancel')
							{
								this._doAction (args.controller, args.contentId, args.actionId, answer == 'yes');
							}
							// else cancel the action
					},
					this
			);
		}
		else
		{
			// Send and validate newsletter
			Ametys.Msg.confirm("{{i18n PLUGINS_NEWSLETTER_WORKFLOW_VALIDATE}}", 
					"{{i18n PLUGINS_NEWSLETTER_WORKFLOW_VALIDATE_AND_SEND_CONFIRM}}", 
					function (answer) {	
						if (answer == 'yes')
						{
							this._doAction (args.controller, args.contentId, args.actionId, true);
						}
						// else cancel the action
					},
					this
			);
		}
 	},
 	
 	/**
 	 * Action performed when the 'OK' button is clicked.
 	 * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller
 	 * @param {String} contentId The content id
 	 * @param {String} actionId The action id
 	 * @param {Boolean} send True if the newsletter has to be sent.
 	 * @private
 	 */
 	_doAction: function (controller, contentId, actionId, send)
 	{
 		var params = {
 			contentId: contentId, 
 			id: contentId, 
 			'with-contents': '/_plugins/cms/contents/info'
 		};
 		if (send)
		{
			params['send'] = true;
		}
		
		Ametys.data.ServerComm.send({
 			plugin: 'cms',
 			url: 'do-action/' + actionId,
 			parameters: params,
 			priority: Ametys.data.ServerComm.PRIORITY_MAJOR,
 			callback: {
 				handler: this._doActionCallback,
 				scope: this,
 				arguments: [contentId, controller]
 			},
 			responseType: 'xml'
 		});
 		
 		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.WORKFLOW_CHANGING,
			targets: {
			    id: Ametys.message.MessageTarget.CONTENT,
				parameters: {
					ids: [contentId]
				}
			}
		});
 	},
 	
 	/**
 	 * Callback function called after #_doAction is processed.
 	 * @param {Object} response The XML response provided by the {@link Ametys.data.ServerComm}
 	 * @param {Array} args The callback parameters passed to the {@link Ametys.data.ServerComm#send} method
 	 * @private
 	 */
 	_doActionCallback: function(response, args)
 	{
 		var contentId = args[0];
 		var controller = args[1];
 		
 		Ext.create("Ametys.message.Message", {
			type: controller.getInitialConfig('workflow-changed-message-type') || Ametys.message.Message.WORKFLOW_CHANGED,
			
			targets: {
			    id: Ametys.message.MessageTarget.CONTENT,
				parameters: {
					ids: [contentId]
				}
			}
		});
 	}
 });