/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.Request;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.Content;
import org.ametys.cms.rights.ContentRightAssignmentContext;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.plugins.newsletter.category.Category;
import org.ametys.plugins.newsletter.category.CategoryProvider;
import org.ametys.plugins.newsletter.category.CategoryProviderExtensionPoint;
import org.ametys.plugins.newsletter.workflow.SendMailEngine;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.content.ModifiableWebContent;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.content.jcr.DefaultWebContent;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

import com.google.common.collect.ImmutableMap;


/**
 * DAO for manipulating newsletter
 *
 */
public class NewsletterDAO extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** The Avalon role */
    public static final String ROLE = NewsletterDAO.class.getName();
    
    /** Right to send a test newsletter */
    public static final String __SEND_TESTING_RIGHT = "Plugins_Newsletter_Right_TestSending";
    
    /** Newsletter content type */
    public static final String __NEWSLETTER_CONTENT_TYPE = "org.ametys.plugins.newsletter.Content.newsletter";
    
    /** Metadata test-unique-id */
    public static final String __TEST_UNIQUE_ID_METADATA = "test-unique-id";

    private AmetysObjectResolver _resolver;
    private CurrentUserProvider _currentUserProvider;
    private RenderingContextHandler _renderingContextHandler;
    private SourceResolver _sourceResolver;
    private CategoryProviderExtensionPoint _categoryProviderEP;
    private Context _context;
    private SiteManager _siteManager;
    private Repository _repository;
    private I18nUtils _i18nUtils;
    private UserLanguagesManager _userLanguagesManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _renderingContextHandler = (RenderingContextHandler) smanager.lookup(RenderingContextHandler.ROLE);
        _sourceResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _repository = (Repository) smanager.lookup(Repository.class.getName());
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _userLanguagesManager = (UserLanguagesManager) smanager.lookup(UserLanguagesManager.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Determines if the newsletter was already sent
     * @param newsletterId the id of newsletter
     * @return true if the newsletter was already sent
     */
    // This callable is only use before validation to check if the newsletter was already sent
    @Callable(rights = "Plugins_Newsletter_Right_ValidateNewsletters", paramIndex = 0, rightContext = ContentRightAssignmentContext.ID)
    public boolean isSent (String newsletterId)
    {
        Content content = _resolver.resolveById(newsletterId);
        return content.getInternalDataHolder().getValue("sent", false);
    }
    
    /**
     * Gets newsletter's properties to JSON format
     * @param newsletter The newsletter
     * @return The newsletter's properties
     */
    public Map<String, Object> getNewsletterProperties(Content newsletter)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", newsletter.getId());
        infos.put("title", newsletter.getTitle());
        infos.put("name", newsletter.getName());
        infos.put("automatic", newsletter.getInternalDataHolder().getValue("automatic", false));
        
        return infos;
    }
    
    /**
     * Send the newsletter to a single recipient, while ignoring the subscribers or the workflow state
     * @param newsletterId The newsletter id
     * @param recipientEmail The recipient
     * @return True if the newsletter was sent
     * @throws IllegalAccessException If a user tried to send a newsletter with insufficient rights
     */
    @Callable(rights = __SEND_TESTING_RIGHT) // This right is assigned on general context, not per newsletter
    public boolean sendTestNewsletter(String newsletterId, String recipientEmail) throws IllegalAccessException
    {
        ModifiableWebContent content = _resolver.resolveById(newsletterId);

        if (!(content instanceof DefaultWebContent))
        {
            throw new UnknownAmetysObjectException("Unable to send newsletter, invalid newsletter id provider '" + newsletterId + "'");
        }
        
        getLogger().info("The user {} sent the newsletter {} to {}", _currentUserProvider.getUser(), newsletterId, recipientEmail);
        
        String uid;
        ModifiableModelLessDataHolder internalDataHolder = content.getInternalDataHolder();
        if (!internalDataHolder.hasValue(__TEST_UNIQUE_ID_METADATA))
        {
            uid = UUID.randomUUID().toString();
            internalDataHolder.setValue(__TEST_UNIQUE_ID_METADATA, uid);
            content.saveChanges();
        }
        else
        {
            uid = internalDataHolder.getValue(__TEST_UNIQUE_ID_METADATA, null);
        }
        
        String siteName = (String) ContextHelper.getRequest(_context).getAttribute("siteName");
        Site site = _siteManager.getSite(siteName);
        boolean includeImages = site.getValue("newsletter-mail-include-images", false, false);
        
        String dataHolderUid = null;
        if (!includeImages && uid != null)
        {
            // create or update temporary content to serve images on the live workspaces
            dataHolderUid = _useDataHolderContent(site, content, uid);
        }
        
        try
        {
            sendNewsletter((DefaultWebContent) content, ImmutableMap.of(recipientEmail, "#token#"), dataHolderUid);
        }
        catch (IOException e)
        {
            getLogger().error("Unable to send the newsletter", e);
            return false;
        }
        
        return true;
    }
    
    private String _useDataHolderContent(Site site, WebContent realContent, String uid)
    {
        try
        {
            RichText richText = realContent.getValue("content");
            
            if (!richText.getAttachmentNames().isEmpty())
            {
                return _createDataHolderContent(site, realContent, uid, richText);
            }
        }
        catch (RepositoryException | IOException e)
        {
            getLogger().error("A repository error occurred when creating the data holder temporary content, when sending a test newsletter", e);
        }
        
        return null;
    }

    private String _createDataHolderContent(Site site, WebContent realContent, String uid, RichText richText) throws RepositoryException, IOException
    {
        Session liveSession = null;
        try
        {
            liveSession = _repository.login(WebConstants.LIVE_WORKSPACE);
            ModifiableTraversableAmetysObject siteContents = _resolver.resolveByPath(site.getPath() + "/" + RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":contents");
            ModifiableTraversableAmetysObject liveSiteContents = _resolver.resolveById(siteContents.getId(), liveSession);
            
            String contentName = realContent.getName() + "-test-" + uid;
            ModifiableWebContent dataHolderContent = null;
            
            if (!liveSiteContents.hasChild(contentName))
            {
                dataHolderContent = liveSiteContents.createChild(contentName, RepositoryConstants.NAMESPACE_PREFIX + ":defaultWebContent");
                dataHolderContent.setTypes(new String[] {__NEWSLETTER_CONTENT_TYPE});
                dataHolderContent.setTitle(realContent.getTitle());
                dataHolderContent.setSiteName(realContent.getSiteName());
                dataHolderContent.setLanguage(realContent.getLanguage());
                dataHolderContent.setLastModified(realContent.getLastModified());
            }
            else
            {
                dataHolderContent = liveSiteContents.getChild(contentName);
            }
            
            richText.setInputStream(new ByteArrayInputStream("unused".getBytes(StandardCharsets.UTF_8)));
            dataHolderContent.setValue("content", richText);
            
            dataHolderContent.saveChanges();
            
            return dataHolderContent.getId();
        }
        finally
        {
            if (liveSession != null)
            {
                liveSession.logout();
            }
        }
    }
    
    /**
     * Send the newsletter to the recipients
     * @param content The newsletter
     * @param recipients The recipients of the newsletter
     * @throws IOException If an error occurred
     */
    public void sendNewsletter(DefaultWebContent content, Map<String, String> recipients) throws IOException
    {
        sendNewsletter(content, recipients, null);
    }
    
    /**
     * Send the newsletter to the recipients
     * @param content The newsletter
     * @param recipients The recipients of the newsletter
     * @param dataHolderId The content to use as a data holder proxy for images. Can be null
     * @throws IOException If an error occurred
     */
    public void sendNewsletter(DefaultWebContent content, Map<String, String> recipients, String dataHolderId) throws IOException
    {
        String language = StringUtils.defaultIfBlank(content.getLanguage(), _userLanguagesManager.getDefaultLanguage());
        String subject = _getSubject(content, language);
        String htmlBody = _getBodyAsHtml(content, dataHolderId, language);
        String textBody = _getBodyAsText(content, dataHolderId, language);
        
        Site site = content.getSite();

        String sender = site.getValue("newsletter-mail-sender");
        
        // Send the mail
        SendMailEngine sendEngine = new SendMailEngine();
        sendEngine.parameterize(subject, htmlBody, textBody, recipients, sender);
        
        new Thread(sendEngine).start();
    }

    /**
     * Get the newsletter mail subject
     * @param content The content
     * @param language The language to use
     * @return The subject
     */
    protected String _getSubject (DefaultWebContent content, String language)
    {
        List<String> i18nparam = new ArrayList<>();
        i18nparam.add(content.getSite().getTitle()); // {0} site
        i18nparam.add(content.getTitle()); // {1} title
        i18nparam.add(String.valueOf(content.getValue("newsletter-number", false, 0L))); // {2} number

        String categoryId = content.getInternalDataHolder().getValue("category");
        Category category = getCategory(categoryId);
        i18nparam.add(category != null ? category.getTitle().getLabel() : StringUtils.EMPTY); // {3} category
        
        I18nizableText i18nTextSubject = new I18nizableText("plugin.newsletter", "PLUGINS_NEWSLETTER_SEND_MAIL_SUBJECT", i18nparam);
        return _i18nUtils.translate(i18nTextSubject, language);
    }
    
    /**
     * Get the newsletter HTML body
     * @param content The content
     * @param dataHolderId The data holder content to use as proxy images
     * @param language The language to use
     * @return The body
     * @throws IOException if an I/O error occurred
     */
    protected String _getBodyAsHtml (DefaultWebContent content, String dataHolderId, String language) throws IOException
    {
        SitemapSource src = null;
        Request request = ContextHelper.getRequest(_context);
        
        Site site = content.getSite();
        boolean includeImages = site.getValue("newsletter-mail-include-images", false, false);
        
        if (includeImages)
        {
            request.setAttribute("forceBase64Encoding", true);
        }
        
        RenderingContext renderingContext = _renderingContextHandler.getRenderingContext();
        try
        {
            _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
        
            String uri = "cocoon://_content.mail?contentId=" + content.getId() + "&site=" + content.getSiteName() + "&lang=" + language + "&_contextPath=" + content.getSite().getUrl();
            if (StringUtils.isNotEmpty(dataHolderId))
            {
                uri += "&useDataHolderContent=" + dataHolderId;
            }
            src = (SitemapSource) _sourceResolver.resolveURI(uri);
            Reader reader = new InputStreamReader(src.getInputStream(), "UTF-8");
            return IOUtils.toString(reader);
        }
        finally
        {
            _sourceResolver.release(src);
            _renderingContextHandler.setRenderingContext(renderingContext);
            request.removeAttribute("forceBase64Encoding");
        }
    }
    
    /**
     * Get the newsletter text body
     * @param content The content
     * @param dataHolderId The data holder content to use as proxy images
     * @param language The language to use
     * @return The body
     * @throws IOException if an I/O error occurred
     */
    protected String _getBodyAsText (DefaultWebContent content, String dataHolderId, String language) throws IOException
    {
        SitemapSource src = null;
        
        RenderingContext renderingContext = _renderingContextHandler.getRenderingContext();
        try
        {
            _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
            
            String uri = "cocoon://_content.text?contentId=" + content.getId() + "&site=" + content.getSiteName() + "&lang=" + language + "&_contextPath=" + content.getSite().getUrl();
            if (StringUtils.isNotEmpty(dataHolderId))
            {
                uri += "&useDataHolderContent=" + dataHolderId;
            }
            src = (SitemapSource) _sourceResolver.resolveURI(uri);
            Reader reader = new InputStreamReader(src.getInputStream(), "UTF-8");
            return IOUtils.toString(reader);
        }
        finally
        {
            _sourceResolver.release(src);
            _renderingContextHandler.setRenderingContext(renderingContext);
        }
    }
    
    /**
     * Get a category
     * @param categoryID The category id
     * @return The category
     */
    public Category getCategory (String categoryID)
    {
        Set<String> ids = _categoryProviderEP.getExtensionsIds();
        for (String id : ids)
        {
            CategoryProvider provider = _categoryProviderEP.getExtension(id);
            if (!categoryID.startsWith("provider_") && provider.hasCategory(categoryID))
            {
                return provider.getCategory(categoryID);
            }
        }
        
        return null;
    }
    

    /**
     * Remove the test newsletter if it exists in live workspace
     * @param content The content
     * @param site The site of the content
     * @throws RepositoryException If an error occurred
     */
    public void removeTestNewsletter(WebContent content, Site site) throws RepositoryException
    {
        if (content.getInternalDataHolder().hasValue(__TEST_UNIQUE_ID_METADATA))
        {
            Session liveSession = null;
            try
            {
                String testUniqueId = content.getInternalDataHolder().getValue(__TEST_UNIQUE_ID_METADATA);
                liveSession = _repository.login(WebConstants.LIVE_WORKSPACE);
                ModifiableTraversableAmetysObject siteContents = _resolver.resolveByPath(site.getPath() + "/" + RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":contents");
                ModifiableTraversableAmetysObject liveSiteContents = _resolver.resolveById(siteContents.getId(), liveSession);
                
                String contentName = content.getName() + "-test-" + testUniqueId;
                
                if (liveSiteContents.hasChild(contentName))
                {
                    AmetysObject child = liveSiteContents.getChild(contentName);
                    if (child instanceof RemovableAmetysObject)
                    {
                        ((RemovableAmetysObject) child).remove();
                        liveSiteContents.saveChanges();
                    }
                }
                
                content.getInternalDataHolder().removeValue(__TEST_UNIQUE_ID_METADATA);
            }
            finally
            {
                if (liveSession != null)
                {
                    liveSession.logout();
                }
            }
        }
    }
}
