/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.auto;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.commons.lang3.StringUtils;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Static implementation of {@link AutomaticNewsletter}.
 */
public class StaticAutomaticNewsletter extends AbstractLogEnabled implements AutomaticNewsletter, Configurable, PluginAware
{
    
    /** The label. */
    protected I18nizableText _label;
    
    /** The description. */
    protected I18nizableText _description;
    
    /** The newsletter title. */
    protected I18nizableText _newsletterTitle;
    
    /** The frequency type. */
    protected FrequencyType _frequencyType;
    
    /** The day numbers. */
    protected List<Integer> _dayNumbers;
    
    /** The filters. */
    protected Map<String, String> _filters;
    
    /** The plugin name. */
    protected String _pluginName;
    
    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _label = _parseI18nizableText(configuration, "label");
        _description = _parseI18nizableText(configuration, "description");
        _newsletterTitle = _parseI18nizableText(configuration, "newsletter-title");
        
        Configuration frequencyConf = configuration.getChild("frequency");
        String frequencyType = frequencyConf.getAttribute("type");
        String dayNumbersStr = frequencyConf.getValue();
        
        _frequencyType = FrequencyType.valueOf(frequencyType.toUpperCase());
        _dayNumbers = new ArrayList<>();
        for (String dayNumber : StringUtils.split(dayNumbersStr, ", "))
        {
            try
            {
                int number = Integer.parseInt(dayNumber);
                _dayNumbers.add(number);
            }
            catch (NumberFormatException e)
            {
                throw new ConfigurationException("Error parsing the day numbers.", configuration, e);
            }
        }
        
        _filters = new HashMap<>();
        Configuration[] filterConfigurations = configuration.getChild("filters").getChildren("filter");
        for (Configuration filterConf : filterConfigurations)
        {
            String name = filterConf.getAttribute("name");
            String id = filterConf.getAttribute("id");
            
            _filters.put(name, id);
        }
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return _label;
    }
    
    @Override
    public I18nizableText getDescription()
    {
        return _description;
    }
    
    @Override
    public I18nizableText getNewsletterTitle()
    {
        return _newsletterTitle;
    }
    
    @Override
    public FrequencyType getFrequencyType()
    {
        return _frequencyType;
    }
    
    @Override
    public Collection<Integer> getDayNumbers()
    {
        return Collections.unmodifiableList(_dayNumbers);
    }
    
    @Override
    public Map<String, String> getFilters()
    {
        return Collections.unmodifiableMap(_filters);
    }
    
    /**
     * Parse an i18n text.
     * @param config the configuration to use.
     * @param name the child name.
     * @return the i18n text.
     * @throws ConfigurationException if the configuration is not valid.
     */
    protected I18nizableText _parseI18nizableText(Configuration config, String name) throws ConfigurationException
    {
        Configuration textConfig = config.getChild(name);
        boolean i18nSupported = textConfig.getAttributeAsBoolean("i18n", false);
        String text = textConfig.getValue("");
        
        if (i18nSupported)
        {
            String catalogue = textConfig.getAttribute("catalogue", null);
            
            if (catalogue == null)
            {
                catalogue = "plugin." + _pluginName;
            }
            
            return new I18nizableText(catalogue, text);
        }
        else
        {
            return new I18nizableText(text);
        }
    }
    
}
