/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.newsletter.category;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.page.jcr.DefaultPage;
import org.ametys.web.tags.TagExpression;

/**
 * This class provides newsletter categories from the site map
 *
 */
public class SitemapCategoryProvider implements LogEnabled, CategoryProvider, Serviceable, Configurable, PluginAware
{
    /** The root id */
    public static final String TAG_NAME = "NEWSLETTER_CATEGORY";
    /** The metadata name for newsletter template */
    public static final String METADATA_TEMPLATE = "newsletterTemplate";
    /** The metadata name for automatic newsletter ids. */
    public static final String METADATA_AUTOMATIC_IDS = "newsletterAutomaticIds";
    /** The metadata name for newsletter template */
    public static final String CATEGORY_PREFIX_ID = "category_";
    
    /** The id */
    protected String _id;
    /** The label */
    protected I18nizableText _label;
    /** The description */
    protected I18nizableText _description;
    /** The plugin name */
    protected String _pluginName;
    /** The feature name */
    protected String _featureName;

    /** The Logger */
    protected Logger _logger;
    
    private AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _label = configureLabel(configuration);
        _description = configureDescription(configuration);
    }
    
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }
    
    @Override
    public boolean isWritable()
    {
        return false;
    }
    
    @Override
    public List<Category> getCategories(String siteName, String lang)
    {
        TagExpression expr = new TagExpression(Operator.EQ, TAG_NAME);
        String xpathQuery = PageQueryHelper.getPageXPathQuery(siteName, lang, "", expr, null);
        
        List<Category> categories = new ArrayList<>();
        
        AmetysObjectIterable<DefaultPage> pages = _resolver.query(xpathQuery);
        for (DefaultPage page : pages)
        {
            Category category = new Category(CATEGORY_PREFIX_ID + page.getId(), page.getName(), _id, new I18nizableText(page.getTitle()), null, page.getValue(METADATA_TEMPLATE), siteName, lang);
            categories.add(category);
        }
        
        return categories;
    }
    
    @Override
    public Collection<Category> getAllCategories(String siteName, String lang)
    {
        return getCategories(siteName, lang);
    }
    
    @Override
    public List<Category> getCategories(String categoryID)
    {
        return new ArrayList<>();
    }

    @Override
    public boolean hasCategory(String categoryID)
    {
        if (categoryID.startsWith(CATEGORY_PREFIX_ID))
        {
            String pageID = categoryID.substring(CATEGORY_PREFIX_ID.length());
            try
            {
                DefaultPage page = _resolver.resolveById(pageID);
                if (page != null)
                {
                    return page.getTags().contains(TAG_NAME);
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // Page does not exist anymore
                return false;
            }
        }
        return false;
    }
    
    @Override
    public Category getCategory(String categoryID)
    {
        if (categoryID.startsWith("category_"))
        {
            String pageID = categoryID.substring(CATEGORY_PREFIX_ID.length());
            DefaultPage page = _resolver.resolveById(pageID);
            if (page != null && page.getTags().contains(TAG_NAME))
            {
                return new Category(CATEGORY_PREFIX_ID + page.getId(), page.getName(), "provider_" + _id, new I18nizableText(page.getTitle()), new I18nizableText(""), page.getValue(METADATA_TEMPLATE), page.getSiteName(), page.getSitemapName());
            }
        }
        return null;
    }
    
    @Override
    public void setTemplate(Category category, String templateName)
    {
        String categoryID = category.getId();
        if (categoryID.startsWith("category_"))
        {
            String pageID = categoryID.substring(CATEGORY_PREFIX_ID.length());
            ModifiablePage page = _resolver.resolveById(pageID);
            if (page != null)
            {
                page.setValue(METADATA_TEMPLATE, templateName);
                page.saveChanges();
            }
        }
    }
    
    @Override
    public Collection<String> getAutomaticIds(String categoryId)
    {
        if (categoryId.startsWith("category_"))
        {
            String pageID = categoryId.substring(CATEGORY_PREFIX_ID.length());
            Page page = _resolver.resolveById(pageID);
            if (page != null)
            {
                String[] autoIds = page.getValue(METADATA_AUTOMATIC_IDS, new String[0]);
                
                return Arrays.asList(autoIds);
            }
        }
        
        // Either the category ID is not a page category, or the page doesn't exist.
        throw new IllegalArgumentException("The provided category ID is invalid.");
    }
    
    @Override
    public void setAutomatic(String categoryId, Collection<String> automaticNewsletterIds)
    {
        if (categoryId.startsWith("category_"))
        {
            String pageID = categoryId.substring(CATEGORY_PREFIX_ID.length());
            Page page = _resolver.resolveById(pageID);
            if (page != null && page instanceof ModifiablePage)
            {
                ModifiablePage modifiablePage = (ModifiablePage) page;
                
                // Convert to array.
                String[] autoIdsArray = automaticNewsletterIds.toArray(new String[automaticNewsletterIds.size()]);
                                
                modifiablePage.setValue(METADATA_AUTOMATIC_IDS, autoIdsArray);
                modifiablePage.saveChanges();
                
                return;
            }
        }
        
        // Either the category ID is not a page category, or the page doesn't exist.
        throw new IllegalArgumentException("The provided category ID is invalid.");
    }
    
    @Override
    public AmetysObjectIterable<Content> getNewsletters(String categoryID, String siteName, String lang)
    {
        Expression cTypeExpr = new ContentTypeExpression(Operator.EQ, "org.ametys.plugins.newsletter.Content.newsletter");
        Expression siteExpr = new StringExpression("site", Operator.EQ, siteName);
        Expression catExpr = new StringExpression("category", Operator.EQ, categoryID, ExpressionContext.newInstance().withInternal(true));
        Expression expr = new AndExpression(cTypeExpr, siteExpr, catExpr);
        
        String xpathQuery = org.ametys.plugins.repository.query.QueryHelper.getXPathQuery(null, "ametys:content", expr);
        return _resolver.query(xpathQuery);
    }
    
    @Override
    public boolean hasChildren(String categoryID)
    {
        return false;
    }

    @Override
    public boolean hasNewsletters(String categoryID, String siteName, String lang)
    {
        Expression expr = new ContentTypeExpression(Operator.EQ, "org.ametys.plugins.newsletter.Content.newsletter");
        Expression siteExpr = new StringExpression("site", Operator.EQ, siteName);
        expr = new AndExpression(expr, siteExpr);
        Expression catExpr = new StringExpression("category", Operator.EQ, categoryID, ExpressionContext.newInstance().withInternal(true));
        expr = new AndExpression(expr, catExpr);
        
        String xpathQuery = org.ametys.plugins.repository.query.QueryHelper.getXPathQuery(null, "ametys:content", expr);
        
        return _resolver.query(xpathQuery).iterator().hasNext();
    }
    
    /**
     * Configure label from the passed configuration
     * @param configuration The configuration
     * @return The label
     * @throws ConfigurationException If an error occurred
     */
    protected I18nizableText configureLabel (Configuration configuration) throws ConfigurationException
    {
        Configuration labelConfiguration = configuration.getChild("label");
        
        if (labelConfiguration.getAttributeAsBoolean("i18n", false))
        {
            return new I18nizableText("plugin." + _pluginName, labelConfiguration.getValue(""));
        }
        else
        {
            return new I18nizableText(labelConfiguration.getValue(""));
        }
    }
    
    /**
     * Configure description from the passed configuration
     * @param configuration The configuration
     * @return The description
     * @throws ConfigurationException If an error occurred
     */
    protected I18nizableText configureDescription (Configuration configuration) throws ConfigurationException
    {
        Configuration descConfiguration = configuration.getChild("description");
        
        if (descConfiguration.getAttributeAsBoolean("i18n", false))
        {
            return new I18nizableText("plugin." + _pluginName, descConfiguration.getValue(""));
        }
        else
        {
            return new I18nizableText(descConfiguration.getValue(""));
        }
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
        _featureName = featureName;
        _id = id;
    }
    
    /**
     * Get the plugin name
     * @return the plugin name
     */
    public String getPluginName()
    {
        return _pluginName;
    }

    @Override
    public I18nizableText getDescription()
    {
        return _description;
    }

    @Override
    public String getId()
    {
        return _id;
    }

    @Override
    public I18nizableText getLabel()
    {
        return _label;
    }

    @Override
    public String getRootId(String siteName, String lang)
    {
        return _id;
    }
}
