/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.nextcloud;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.AbstractGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.resources.generators.ResourcesExplorerGenerator;
import org.ametys.plugins.extrausermgt.oauth.OAuthProvider;
import org.ametys.plugins.extrausermgt.oauth.OauthProviderExtensionPoint;
import org.ametys.runtime.config.Config;

import com.github.sardine.DavResource;
import com.github.sardine.Sardine;
import com.github.sardine.impl.SardineImpl;
import com.nimbusds.oauth2.sdk.token.AccessToken;

/**
 * Generate SAX event to represent the content of a remote Nextcloud repository.
 * 
 * This generator is intended to produced an XML equivalent to one produced by {@link ResourcesExplorerGenerator}.
 * The main difference will be in the attribute available. No id or icon cls can be computed.
 */
public class NextcloudExplorerGenerator extends AbstractGenerator implements Serviceable, Initializable
{
    private String _remoteUrl;
    private OauthProviderExtensionPoint _oauthEP;

    public void service(ServiceManager manager) throws ServiceException
    {
        _oauthEP = (OauthProviderExtensionPoint) manager.lookup(OauthProviderExtensionPoint.ROLE);
    }

    public void initialize() throws Exception
    {
        _remoteUrl = Config.getInstance().getValue("nextcloud.remote.url");
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        OAuthProvider nextcloudProvider = _oauthEP.getExtension("nextcloud");
        Optional<AccessToken> accessToken = nextcloudProvider.getStoredAccessToken();
        Optional<String> userId = nextcloudProvider.getStoredCustomParameter("user_id");
        if (accessToken.isEmpty() || userId.isEmpty())
        {
            contentHandler.startDocument();
            XMLUtils.createElement(contentHandler, "not-connected");
            contentHandler.endDocument();
            return;
        }
        
        Sardine webDavClient = new SardineImpl(accessToken.get().getValue());
        
        contentHandler.startDocument();
        
        saxRemoteFolder("/remote.php/dav/files/" + userId.get() + "/", webDavClient);
        
        contentHandler.endDocument();
    }

    private void saxRemoteFolder(String path, Sardine davClient) throws ProcessingException, SAXException
    {
        try
        {
            // Retrieve all resources
            List<DavResource> resources = davClient.list(_remoteUrl + path);
            DavResource target = null;
            for (Iterator iterator = resources.iterator(); iterator.hasNext() && target == null;)
            {
                DavResource resource = (DavResource) iterator.next();
                if (StringUtils.equals(resource.getPath(), path))
                {
                    resources.remove(resource);
                    target = resource;
                }
            }
            if (target != null)
            {
                saxDirectory(target, resources, davClient);
            }
            else
            {
                throw new ProcessingException("Failed to retrieve information for remote folder at path " + path);
            }
        }
        catch (IOException e)
        {
            throw new ProcessingException("Failed to list content of remote folder '" + path + "'", e);
        }
        
    }

    private void saxDirectory(DavResource directory, List<DavResource> children, Sardine davClient) throws SAXException, ProcessingException
    {
        AttributesImpl childAtts = new AttributesImpl();
        
        childAtts.addCDATAAttribute("name", directory.getName());
        
        childAtts.addCDATAAttribute("path", directory.getPath());
        childAtts.addCDATAAttribute("type", ResourcesExplorerGenerator.RESOURCE_COLLECTION);
        
        boolean hasResources = false;
        boolean hasChildNodes = false;
        for (Iterator iterator = children.iterator(); iterator.hasNext() && !(hasChildNodes && hasResources);)
        {
            DavResource davResource = (DavResource) iterator.next();
            if (davResource.isDirectory())
            {
                hasChildNodes = true;
            }
            else
            {
                hasResources = true;
            }
        }
        if (hasChildNodes)
        {
            childAtts.addCDATAAttribute("hasChildNodes", "true");
        }
        
        if (hasResources)
        {
            childAtts.addCDATAAttribute("hasResources", "true");
        }
        XMLUtils.startElement(contentHandler, "Node", childAtts);
        
        for (DavResource child : children)
        {
            if (child.isDirectory())
            {
                saxRemoteFolder(child.getPath(), davClient);
            }
            else
            {
                saxResource(child);
            }
        }
        
        XMLUtils.endElement(contentHandler, "Node");
        
    }

    private void saxResource(DavResource resource) throws SAXException
    {
        AttributesImpl childAtts = new AttributesImpl();
        childAtts.addCDATAAttribute("href", _remoteUrl + resource.getHref().toString());
        childAtts.addCDATAAttribute("name", resource.getName());
        childAtts.addCDATAAttribute("mimetype", resource.getContentType());
        childAtts.addCDATAAttribute("lastModified", DateUtils.dateToString(resource.getModified()));
        childAtts.addCDATAAttribute("size", String.valueOf(resource.getContentLength()));
        childAtts.addCDATAAttribute("type", ResourcesExplorerGenerator.RESOURCE);
        childAtts.addCDATAAttribute("path", resource.getPath());
        
        XMLUtils.createElement(contentHandler, "Node", childAtts);
    }
}
