/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.notification;

import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.SubscriptionFactory;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.pagesubscription.type.TagSubscriptionType;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityFactory;
import org.ametys.plugins.repository.activities.ActivityHelper;
import org.ametys.plugins.repository.activities.ActivityTypeExtensionPoint;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.activities.AbstractSiteAwareActivityType;
import org.ametys.web.activities.PageUpdatedActivityType;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Helper for tag notifications
 */
public class TagNotificationsHelper extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The avalon role */
    public static final String ROLE = TagNotificationsHelper.class.getName();
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The tga provider extension point */
    protected TagProviderExtensionPoint _tagProviderEP;
    
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;

    /** The activity type extension point */
    protected ActivityTypeExtensionPoint _activityTypeEP;

    /** The right manager */
    protected RightManager _rightManager;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) smanager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _tagProviderEP = (TagProviderExtensionPoint) smanager.lookup(TagProviderExtensionPoint.ROLE);
        _cacheManager = (AbstractCacheManager) smanager.lookup(AbstractCacheManager.ROLE);
        _activityTypeEP = (ActivityTypeExtensionPoint) smanager.lookup(ActivityTypeExtensionPoint.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createMemoryCache(ROLE, 
                new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_TAG_CACHE"),
                new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_TAG_CACHE_DESCRIPTION"),
                true,
                null);
    }
    
    /**
     * Get the tag content notifications for each subscription of the current user
     * @param siteName the site name
     * @return the content notifications
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public List<Map<String, Object>> getUserContentNotifications(String siteName)
    {
        List<Map<String, Object>> notifications2json = new ArrayList<>();
        
        Site site = _siteManager.getSite(siteName);
        
        UserIdentity user = _currentUserProvider.getUser();
        TagSubscriptionType tagSubscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);
        
        List<Subscription> subscriptions = tagSubscriptionType.getUserSubscriptions(site, null, BroadcastChannel.SITE, user, false, null);
        for (Subscription subscription : subscriptions)
        {
            Map<String, Object> subscriptionAsJson = tagSubscriptionType.subscriptionToJSON(subscription);
            
            Map<Content, ZonedDateTime> contents = getUpdatedContents(subscription);
            if (!contents.isEmpty())
            {
                List<Map<String, String>> contentsAsJson = contents.keySet()
                        .stream()
                        .map(c -> Map.of("id", c.getId(), "title", c.getTitle()))
                        .toList();
                
                subscriptionAsJson.put("contents", contentsAsJson);            

                ZonedDateTime lastActivityDate = subscription.getFrequency() != Frequency.INSTANT
                    ? _getNotificationDate(subscription)
                    : contents.values()
                        .stream()
                        .max(ZonedDateTime::compareTo)
                        .get();
                
                subscriptionAsJson.put("lastActivityDate", lastActivityDate);
                
                ZonedDateTime lastReadDate = tagSubscriptionType.getLastReadDate(subscription, user);
                boolean hasRead = lastReadDate != null && lastReadDate.isAfter(lastActivityDate);
                subscriptionAsJson.put("hasRead", hasRead);
            }
            
            notifications2json.add(subscriptionAsJson);
        }
        
        return notifications2json;
    }
    
    /**
     * Get contents to be notified from a subscription
     * @param subscription the subscription
     * @return the map with the content and its last activity date
     */
    public Map<Content, ZonedDateTime> getUpdatedContents(Subscription subscription)
    {
        String siteName = subscription.getSite().getName();
        String tagName = subscription.getValue(TagSubscriptionType.TAG);
        
        Frequency frequency = subscription.getFrequency();
        ZonedDateTime notificationDate = _getNotificationDate(subscription);
        
        TagSubscriptionKey key = TagSubscriptionKey.of(siteName, tagName, frequency, notificationDate);
        Map<String, ZonedDateTime> contents = _getCache().get(key, k -> _getUpdatedContents(siteName, tagName, frequency, notificationDate));
        
        return contents.keySet()
            .stream()
            .map(this::_resolveSilently)
            .filter(Objects::nonNull)
            .filter(c -> _rightManager.currentUserHasReadAccess(c))
            .collect(Collectors.toMap(
                c -> c, 
                c -> contents.get(c.getId()),
                (e1, e2) -> e2,
                LinkedHashMap::new
            ));
    }
    
    /**
     * Get contents to be notified from a subscription for the current user
     * @param siteName the site name
     * @param tagName the tag name
     * @param frequency the frequency
     * @param timing the timing
     * @return the map with the content and its last activity date
     */
    public Map<Content, ZonedDateTime> getUpdatedContents(String siteName, String tagName, Frequency frequency, FrequencyTiming timing)
    {
        return getUpdatedContents(siteName, tagName, frequency, timing, _currentUserProvider.getUser());
    }
    
    /**
     * Get contents to be notified from a subscription for the given user
     * @param siteName the site name
     * @param tagName the tag name
     * @param frequency the frequency
     * @param timing the timing
     * @param user the user
     * @return the map with the content and its last activity date
     */
    public Map<Content, ZonedDateTime> getUpdatedContents(String siteName, String tagName, Frequency frequency, FrequencyTiming timing, UserIdentity user)
    {
        ZonedDateTime notificationDate = FrequencyHelper.getNotificationDate(frequency, timing);
        
        TagSubscriptionKey key = TagSubscriptionKey.of(siteName, tagName, frequency, notificationDate);
        Map<String, ZonedDateTime> contents = _getCache().get(key, k -> _getUpdatedContents(siteName, tagName, frequency, notificationDate));
        
        return contents.keySet()
            .stream()
            .map(this::_resolveSilently)
            .filter(Objects::nonNull)
            .filter(c -> _rightManager.hasReadAccess(user, c))
            .collect(Collectors.toMap(
                c -> c, 
                c -> contents.get(c.getId()),
                (e1, e2) -> e2,
                LinkedHashMap::new
            ));
    }
    
    private Map<String, ZonedDateTime> _getUpdatedContents(String siteName, String tagName, Frequency frequency, ZonedDateTime notificationDate)
    {
        Map<String, ZonedDateTime> contents = new HashMap<>();
        for (Activity activity : _getContentActivities(siteName, tagName, frequency, notificationDate))
        {
            String contentId = activity.getValue(PageUpdatedActivityType.CONTENT_ID);
            Content content = _resolveSilently(contentId);
            if (content != null)
            {
                ZonedDateTime date = activity.getValue(ActivityFactory.DATE);
                if (contents.containsKey(contentId))
                {
                    ZonedDateTime lastActivityDate = contents.get(contentId);
                    if (lastActivityDate == null || date.isAfter(lastActivityDate))
                    {
                        contents.put(contentId, lastActivityDate);
                    }
                }
                else
                {
                    contents.put(contentId, date);
                }
            }
        }
        
        return contents.entrySet()
            .stream()
            .sorted((e1, e2) -> e2.getValue().compareTo(e1.getValue())) // ordered by last activity date
            .collect(Collectors.toMap(
                e -> e.getKey(), 
                e -> e.getValue(),
                (e1, e2) -> e2,
                LinkedHashMap::new
            ));
    }
    
    private AmetysObjectIterable<Activity> _getContentActivities(String siteName, String tagName, Frequency frequency, ZonedDateTime notificationDate)
    {
        List<Expression> finalExprs = new ArrayList<>();
        
        finalExprs.addAll(FrequencyHelper.getDateExpressions(frequency, notificationDate));
        
        Expression[] activityTypeExpressions = _activityTypeEP.getExtensionsIds()
                .stream()
                .map(_activityTypeEP::getExtension)
                .filter(PageUpdatedActivityType.class::isInstance)
                .map(type -> new StringExpression(ActivityFactory.ACTIVITY_TYPE_ID, Operator.EQ, type.getId()))
                .toArray(Expression[]::new);
        finalExprs.add(new OrExpression(activityTypeExpressions));
        finalExprs.add(new StringExpression(PageUpdatedActivityType.CONTENT_TAGS, Operator.EQ, tagName));
        finalExprs.add(new StringExpression(AbstractSiteAwareActivityType.SITE_NAME, Operator.EQ, siteName));
        
        Expression finalExpr = new AndExpression(finalExprs.toArray(new Expression[finalExprs.size()]));
        
        String xpathQuery = ActivityHelper.getActivityXPathQuery(finalExpr);
        return _resolver.query(xpathQuery);
    }
    
    private ZonedDateTime _getNotificationDate(Subscription subscription)
    {
        // Get frequency and timing of this subscription
        Frequency frequency = subscription.getFrequency();
        String time = subscription.getValue(SubscriptionFactory.FORCED_HOUR, FrequencyHelper.getDefaultFrequencyTime());
        long day = subscription.getValue(SubscriptionFactory.FORCED_DAY, FrequencyHelper.getDefaultFrequencyDay());
        
        FrequencyTiming timing = new FrequencyTiming(day, time);
        
        return FrequencyHelper.getNotificationDate(frequency, timing);
    }
    
    private Content _resolveSilently(String contentId)
    {
        try
        {
            return _resolver.resolveById(contentId);
        }
        catch (Exception e) 
        {
            getLogger().warn("Can't resolve content with id '{}'", contentId, e);
        }
        
        return null;
    }
    
    private Cache<TagSubscriptionKey, Map<String, ZonedDateTime>> _getCache()
    {
        return _cacheManager.get(ROLE);
    }
    
    /**
     * Clear tag cache 
     * @param siteName the site name
     * @param tagName the tag name
     */
    public void clearCache(String siteName, String tagName)
    {
        _getCache().invalidate(TagSubscriptionKey.of(siteName, tagName));
    }
    
    static class TagSubscriptionKey extends AbstractCacheKey
    {
        TagSubscriptionKey(String siteName, String tagName, Frequency frequency, ZonedDateTime notificationDate)
        {
            super(siteName, tagName, frequency, notificationDate);
        }
        
        static TagSubscriptionKey of(String siteName, String tagName, Frequency frequency, ZonedDateTime notificationDate)
        {
            return new TagSubscriptionKey(siteName, tagName, frequency, notificationDate);
        }
        
        static TagSubscriptionKey of(String siteName, String tagName)
        {
            return new TagSubscriptionKey(siteName, tagName, null, null);
        }
    }
}
