/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.pagesubscription.type.SubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.SimpleAmetysObjectFactory;
import org.ametys.web.data.type.ModelItemTypeExtensionPoint;

/**
 * AmetysObject factory associated with {@link Subscription}
 */
public class SubscriptionFactory extends SimpleAmetysObjectFactory
{
    /** Avalon Role for subscription data types */
    public static final String MODEL_ITEM_TYPE_EXTENSION_ROLE = ModelItemTypeExtensionPoint.class.getName() + ".Subscription";
    
    /** Subscription node type name. */
    public static final String NODE_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":subscription";
    
    /** Data name for the subscription type id */
    public static final String SUBSCRIPTION_TYPE_ID = "typeId";
    
    /** Data name for the subscriber */
    public static final String SUBSCRIBER = "subscriber";

    /** Data name for the subscribers group */
    public static final String SUBSCRIBERS_GROUP = "group";

    /** Data name for the date of the subscription */
    public static final String DATE = "date";

    /** Data name for the frequency of the subscription */
    public static final String FREQUENCY = "frequency";

    /** Data name for the broadcast channel of the subscription */
    public static final String BROADCAST_CHANNEL = "broadcastChannel";
    
    /** Data name to determine if the subscription is forced or not */
    public static final String IS_FORCED = "isForced";
    
    /** Data name for the day of the subscription if it's forced */
    public static final String FORCED_DAY = "forcedDay";
    
    /** Data name for the hour of the subscription if it's forced */
    public static final String FORCED_HOUR = "forcedHour";
    
    private ModelItemTypeExtensionPoint _modelItemTypeEP;
    private SubscriptionTypeExtensionPoint _subscriptionTypeEP;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _modelItemTypeEP = (ModelItemTypeExtensionPoint) serviceManager.lookup(MODEL_ITEM_TYPE_EXTENSION_ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) serviceManager.lookup(SubscriptionTypeExtensionPoint.ROLE);
    }
    
    @Override
    public Subscription getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException, RepositoryException
    {
        return new Subscription(node, parentPath, this);
    }

    /**
     * Retrieves the extension point with available modelItem types for {@link Subscription}
     * @return the extension point with available modelItem types for {@link Subscription}
     */
    public ModelItemTypeExtensionPoint getElementTypesExtensionPoint()
    {
        return _modelItemTypeEP;
    }

    /**
     * Retrieve an {@link SubscriptionType}
     * @param subscriptionTypeId the type id
     * @return the subscription type
     */
    public SubscriptionType getSubscriptionType(String subscriptionTypeId)
    {
        return _subscriptionTypeEP.getExtension(subscriptionTypeId);
    }
}
