/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.generator;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.notification.TagNotificationsHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Generator for user tag notifications
 */
public class UserTagNotificationsGenerator extends ServiceableGenerator
{
    /** The notification helper */
    protected TagNotificationsHelper _notificationHelper;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content helper */
    protected ContentHelper _contentHelper;
    
    /** The source resolver */
    protected SourceResolver _srcResolver;
    
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _notificationHelper = (TagNotificationsHelper) smanager.lookup(TagNotificationsHelper.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
        _srcResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _contentTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        String subscriptionId = (String) request.get("subscriptionId");
        String viewName = (String) request.get("viewName");
        
        Subscription subscription = _resolver.resolveById(subscriptionId);
        subscription.getSubscriptionType().markAsRead(subscription, _currentUserProvider.getUser());
        
        XMLUtils.startElement(contentHandler, "contents");
        for (Content content : _notificationHelper.getUpdatedContents(subscription).keySet())
        {
            _saxContent(contentHandler, content, viewName);
        }
        XMLUtils.endElement(contentHandler, "contents");
        
        contentHandler.endDocument();
    }
    
    /**
     * SAX a content in its specific view
     * @param handler The content handler to SAX into
     * @param content The content to SAX
     * @param viewName The view to use
     * @throws SAXException If an error occurs while SAXing
     * @throws IOException If an error occurs while retrieving content.
     */
    protected void _saxContent (ContentHandler handler, Content content, String viewName) throws SAXException, IOException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", content.getId());
        attrs.addCDATAAttribute("name", content.getName());
        attrs.addCDATAAttribute("title", content.getTitle(null));
        attrs.addCDATAAttribute("lastModifiedAt", DateUtils.zonedDateTimeToString(content.getLastModified()));
        
        XMLUtils.startElement(handler, "content", attrs);
        
        String uri = _contentHelper.getContentHtmlViewUrl(content, viewName, Map.of("checkUserAccess", "true"));
        SitemapSource src = null;
        
        try
        {
            src = (SitemapSource) _srcResolver.resolveURI(uri);
            src.toSAX(new IgnoreRootHandler(handler));
        }
        finally
        {
            _srcResolver.release(src);
        }
        
        XMLUtils.endElement(handler, "content");
    }
    
}
