/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.schedulable.tag;

import java.io.IOException;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.quartz.JobExecutionContext;

import org.ametys.cms.repository.Content;
import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.mail.TagSubscriptionSummaryMailBodyHelper;
import org.ametys.plugins.pagesubscription.mail.TagSubscriptionSummaryMailBodyHelper.MailBodyBuilder;
import org.ametys.plugins.pagesubscription.notification.TagNotificationsHelper;
import org.ametys.plugins.pagesubscription.type.SubscriptionType.FrequencyTiming;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.pagesubscription.type.TagSubscriptionType;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizable;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.WebConstants;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Abstract schedulable to send not forced tag notification summary by mail
 * Currently, it's only the custom user thematic subscription
 */
public abstract class AbstractSendTagNotificationSummarySchedulable extends AbstractStaticSchedulable
{
    /** Tag for page that contains subscriptions configuration service */
    public static final String SUBSCRIPTIONS_CONFIG_PAGE_TAG = "TAG_SUBSCRIPTIONS_CONFIG";
    
    /** The mail subject */
    protected I18nizableText _mailSubject;
    
    /** The subscription type EP */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The tag subscription type */
    protected TagSubscriptionType _tagSubscriptionType;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    /** The tag notification helper */
    protected TagNotificationsHelper _tagNotificationHelper;
    
    /** The page DAO */
    protected PageDAO _pageDAO;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _tagSubscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _tagNotificationHelper = (TagNotificationsHelper) manager.lookup(TagNotificationsHelper.ROLE);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        
        _mailSubject = I18nizableText.parseI18nizableText(configuration.getChild("mail-subject"), "plugin." + _pluginName);
    }
    
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        Frequency frequency = getFrequency(context);
        FrequencyTiming timing = getFrequencyTiming(context);
        ZonedDateTime notificationDate = FrequencyHelper.getNotificationDate(frequency, timing);
        
        // For each site
        for (Site site : getSiteToNotify(context))
        {
            Request request = ContextHelper.getRequest(_context);
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, site.getName());
            
            // Retrieve the current context.
            RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
            try
            {
                // Force rendering context.FRONT to resolve URI
                _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
                
                // Get all subscribers by tags
                Map<CMSTag, Set<UserIdentity>> subscribersByTag = getSubscribersByTag(context, site, frequency);
                for (CMSTag tag : subscribersByTag.keySet())
                {
                    for (UserIdentity subscriber : subscribersByTag.get(tag))
                    {
                        // Get all updated content for a given tag in the schedulable frequency
                        // Timing is the default configuration timing (day and hour)
                        Map<Content, ZonedDateTime> updatedContents = _tagNotificationHelper.getUpdatedContents(site.getName(), tag.getName(), frequency, timing, subscriber);
                        if (!updatedContents.isEmpty()) // Send mail only if there are some updated contents
                        {
                            Set<Content> contents = updatedContents.keySet();
                            
                            User user = _userManager.getUser(subscriber);
                            if (user != null)
                            {
                                String lang = _getLanguage(user);
                                
                                String subject = _i18nUtils.translate(_getI18nSubject(site, tag), lang);
                                String body = getMailBody(site, tag, contents, frequency, notificationDate, lang);
                            
                                SendMailHelper.newMail()
                                    .withRecipient(user.getEmail())
                                    .withSubject(subject)
                                    .withHTMLBody(body)
                                    .withInlineCSS(false)
                                    .withAsync(true)
                                    .sendMail();
                            }
                        }
                    }
                }
            }
            finally
            {
                // Restore current context
                _renderingContextHandler.setRenderingContext(currentContext);
            }
        }
    }
    
    private String _getLanguage(User user)
    {
        String language = null;
        if (user != null)
        {
            // First try to get the user's language
            language = user.getLanguage();
        }
        
        language = StringUtils.defaultIfBlank(language, _userLanguagesManager.getDefaultLanguage());
        
        return language;
    }
    
    /**
     * Get the site to notify by the schedulable
     * @param context the job context
     * @return the sites
     */
    protected abstract List<Site> getSiteToNotify(JobExecutionContext context);
    
    /**
     * Get the frequency of the schedulable
     * @param context the job context
     * @return the frequency
     */
    protected abstract Frequency getFrequency(JobExecutionContext context);
    
    /**
     * Get the frequency timing of the schedulable
     * @param context the job context
     * @return the frequency timing
     */
    protected abstract FrequencyTiming getFrequencyTiming(JobExecutionContext context);

    /**
     * Get subscribers by tag
     * @param context the job context
     * @param site the site
     * @param frequency the frequency
     * @return the map of tag with its subscribers
     */
    protected abstract Map<CMSTag, Set<UserIdentity>> getSubscribersByTag(JobExecutionContext context, Site site, Frequency frequency);
    
    /**
     * Get the subject of the mail
     * @param site the site
     * @param tag the thematic
     * @return the subject of the mail
     */
    protected I18nizable _getI18nSubject(Site site, CMSTag tag)
    {
        Map<String, I18nizableTextParameter> params = Map.of(
            "tag", tag.getTitle(),
            "site", new I18nizableText(site.getTitle())
        );
        return new I18nizableText(_mailSubject.getCatalogue(), _mailSubject.getKey(), params);
    }
    
    /**
     * Get the mail body of the mail
     * @param site the site
     * @param tag the thematic
     * @param updatedContents the updated contents
     * @param frequency the frequency
     * @param notificationDate the notification date
     * @param lang the language
     * @return the mail body of the mail
     * @throws IOException if an error occurred
     */
    protected String getMailBody(Site site, CMSTag tag, Set<Content> updatedContents, Frequency frequency, ZonedDateTime notificationDate, String lang) throws IOException
    {
        MailBodyBuilder bodyBuilder = TagSubscriptionSummaryMailBodyHelper.newHTMLBody()
            .withLanguage(lang)
            .withTitle(_getMailBodyTitle(tag))
            .withHint(_getMailBodyHint(frequency, notificationDate, lang))
            .withContents(updatedContents)
            .withLink(site.getUrl(), new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_NOTIFICATIONS_MAIL_SITE_LINK"));
        
        Optional<String> pageTagUrl = _pageDAO.findPagedIdsByTag(site.getName(), lang, SUBSCRIPTIONS_CONFIG_PAGE_TAG)
                .stream()
                .map(id -> ResolveURIComponent.resolve("page", id, false, true))
                .findFirst();
        if (pageTagUrl.isPresent())
        {
            bodyBuilder.withFooterLink(pageTagUrl.get(), new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_TAG_NOTIFICATIONS_MAIL_PREFERENCE_LINK"), "core-ui", "img/mail/icon-letter.png");
        }
        
        return bodyBuilder.build();
    }
    
    /**
     * Get the title in mail body
     * @param tag the thematic
     * @return the title
     */
    protected I18nizableText _getMailBodyTitle(CMSTag tag)
    {
        Map<String, I18nizableTextParameter> params = Map.of("tag", tag.getTitle());
        return new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_TAG_NOTIFICATIONS_MAIL_TITLE", params);
    }
    
    /**
     * Get the hint in mail body from the frequency and notification date
     * @param frequency the frequency
     * @param notificationDate the notification date
     * @param lang the manguage
     * @return the hint message
     */
    protected I18nizableText _getMailBodyHint(Frequency frequency, ZonedDateTime notificationDate, String lang)
    {
        String pattern = _i18nUtils.translate(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIPTION_MAIL_DATE_FORMAT"), lang);
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern);
        Map<String, I18nizableTextParameter> params = Map.of("date", new I18nizableText(formatter.withLocale(Locale.of(lang)).format(notificationDate)));
        
        String key = "PLUGINS_PAGE_SUBSCRIPTION_MAIL_TAG_NOTIFICATIONS_MAIL_MESSAGE_" + frequency.name();
        return new I18nizableText("plugin.page-subscription", key, params);
    }
}
