/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

const GroupIdentity = Java.type('org.ametys.core.group.GroupIdentity');
const UserIdentity = Java.type('org.ametys.core.user.UserIdentity');

const VISIBILITY = "ametys-internal:visibility";

const PROFILE_READ_ACCESS = "ametys-internal:read-access";

const PROFILE_WRITE_ACCESS = "ametys-internal:write-access";
    
const USERS = "ametys-internal:users";

const GROUPS = "ametys-internal:groups";

const READER = "READER";
const QUERIESMIGRATIONPROFILELABEL = "Queries migration";
const QUERIESMIGRATIONPROFILEID = "queries-migration";

const TITLE = "ametys-internal:label";
const TYPE = "ametys-internal:type";
const DESCRIPTION = "ametys-internal:description";
const DOCUMENTATION = "ametys-internal:documentation";
const CONTENT = "ametys-internal:content";
const AUTHOR = "ametys-internal:author";
const LASTMODIFICATIONDATE = "ametys-internal:lastModificationDate";

const _profileAssignmentStorageExtensionPoint = Ametys.serviceManager.lookup("org.ametys.core.right.ProfileAssignmentStorageExtensionPoint");
const _rightProfileDAO = Ametys.serviceManager.lookup("org.ametys.core.right.RightProfilesDAO");
// Migrate data, set contributor and creation date, upgrade queries to kernel rights

// Replace queries and query node types to read old attributes
Repository.registerNodeType("queries");
Repository.registerNodeType("query");

let queriesProfile = _rightProfileDAO.getProfile(QUERIESMIGRATIONPROFILEID);
if (queriesProfile == null)
{
    queriesProfile = _rightProfileDAO.addProfile(QUERIESMIGRATIONPROFILELABEL);
}
  
_rightProfileDAO.updateRights(queriesProfile, ["QueriesDirectory_Rights_Admin"]);

let migratedQueries = 0;

try
{
    Repository.query("//element(*, ametys:query)").forEach(function(query)
    {
        let node = query.getNode();
        let hasChanges = false;
        
        // Query is now a ModifiableModelAwareDataAwareAmetysObject, attributes should not be stored as ametys-internal properties
        if (node.hasProperty(LASTMODIFICATIONDATE))
        {
            query.setLastModificationDate(node.getProperty(LASTMODIFICATIONDATE).getValue().getDate().getTime());
            // Set creation date with lastModificationDate attribute (which was not updated anyway, so acted as creation date)
            query.setCreationDate(query.getLastModificationDate());
            node.getProperty(LASTMODIFICATIONDATE).remove();
            hasChanges = true;
        }
        if (node.hasNode(AUTHOR))
        {
            var authorNode = node.getNode(AUTHOR);
            var author = new UserIdentity(authorNode.getProperty("ametys:login").getString(), authorNode.getProperty("ametys:population").getString());
            query.setAuthor(author);
            // Set last contributor with the author
            query.setContributor(query.getAuthor());
            authorNode.remove();
            hasChanges = true;
        }
    
        if (node.hasProperty(TITLE))
        {
            query.setTitle(node.getProperty(TITLE).getValue().getString());
            node.getProperty(TITLE).remove();
            hasChanges = true;
        }
        
        if (node.hasProperty(TYPE))
        {
            query.setType(node.getProperty(TYPE).getValue().getString());
            node.getProperty(TYPE).remove();
            hasChanges = true;
        }
        
        if (node.hasProperty(DESCRIPTION))
        {
            query.setDescription(node.getProperty(DESCRIPTION).getValue().getString());
            node.getProperty(DESCRIPTION).remove();
            hasChanges = true;
        }
        
        if (node.hasProperty(CONTENT))
        {
            query.setContent(node.getProperty(CONTENT).getValue().getString());
            node.getProperty(CONTENT).remove();
            hasChanges = true;
        }
        
        if (node.hasProperty(VISIBILITY))
        {
            var visibility = node.getProperty(VISIBILITY).getValue().getString();
            
            if (visibility == "PUBLIC")
            {
                // grant access to reader profile to every connected users
                _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromAnyConnectedUser(READER, query);
                _profileAssignmentStorageExtensionPoint.allowProfileToAnyConnectedUser(READER, query);
                
                _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromAnyConnectedUser(QUERIESMIGRATIONPROFILEID, query);
                _profileAssignmentStorageExtensionPoint.allowProfileToAnyConnectedUser(QUERIESMIGRATIONPROFILEID, query);
                hasChanges = true;
            }
            else if (visibility == "SHARED")
            {
                if (node.hasNode(PROFILE_READ_ACCESS))
                {
                    var profileNode = node.getNode(PROFILE_READ_ACCESS);
                    
                    if (profileNode.hasNode(USERS))
                    {
                        var usersIterator = profileNode.getNodes(USERS);
                        
                        // grant access to reader profile to each user that had reader right
                        while (usersIterator.hasNext())
                        {
                            var user = usersIterator.next();
                            
                            var userIdentity = new UserIdentity(user.getProperty("ametys:login").getString(), user.getProperty("ametys:population").getString());
                            
                            _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromUser(userIdentity, READER, query);
                            _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, READER, query);
                        }
                        hasChanges = true;
                    }
                    if (profileNode.hasNode(GROUPS))
                    {
                        var groupsIterator = profileNode.getNodes(GROUPS);
                        
                        // grant access to reader profile to each group that had reader right
                        while (groupsIterator.hasNext())
                        {
                            var group = groupsIterator.next();
                            var groupIdentity = new GroupIdentity(group.getProperty("ametys:groupId").getString(), group.getProperty("ametys:groupDirectory").getString());
                            
                            _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromGroup(groupIdentity, READER, query);
                            _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, READER, query);
                        }
                        hasChanges = true;
                    }
                }
    
                if (node.hasNode(PROFILE_WRITE_ACCESS))
                {
                    var profileNode = node.getNode(PROFILE_WRITE_ACCESS);
                    
                    if (profileNode.hasNode(USERS))
                    {
                        var usersIterator = profileNode.getNodes(USERS);
                        
                        // grant access to reader profile to each user that had reader right
                        while (usersIterator.hasNext())
                        {
                            var user = usersIterator.next();
                            
                            var userIdentity = new UserIdentity(user.getProperty("ametys:login").getString(), user.getProperty("ametys:population").getString());
    
                            _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromUser(userIdentity, READER, query);
                            _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, READER, query);
                            
                            _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromUser(userIdentity, QUERIESMIGRATIONPROFILEID, query);
                            _profileAssignmentStorageExtensionPoint.allowProfileToUser(userIdentity, QUERIESMIGRATIONPROFILEID, query);
                        }
                        hasChanges = true;
                    }
                    if (profileNode.hasNode(GROUPS))
                    {  
                        var groupsIterator = profileNode.getNodes(GROUPS);
                        
                        // grant access to reader profile to each group that had reader right
                        while (groupsIterator.hasNext())
                        {
                            var group = groupsIterator.next();
                            var groupIdentity = new GroupIdentity(group.getProperty("ametys:groupId").getString(), group.getProperty("ametys:groupDirectory").getString());
    
                            _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromGroup(groupIdentity, READER, query);
                            _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, READER, query);
                            
                            _profileAssignmentStorageExtensionPoint.removeDeniedProfileFromGroup(groupIdentity, QUERIESMIGRATIONPROFILEID, query);
                            _profileAssignmentStorageExtensionPoint.allowProfileToGroup(groupIdentity, QUERIESMIGRATIONPROFILEID, query);
                        }
                        hasChanges = true;
                    }
                }
            }
        }
    
        if (hasChanges)
        {
            query.saveChanges();
            migratedQueries ++;
        }
    })
}
catch (e)
{
    logger.error("An error has occured during query migration for query '" + query.getId() , e);
    throw e;
} 
finally 
{
    logger.info(migratedQueries + " handled queries");
}
;
