/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Queries-directory-specific content search tool, used to refresh or close tools after interactions with queries directory.
 */
Ext.define('Ametys.plugins.queriesdirectory.tool.ContentSearchTool', {
    override: 'Ametys.plugins.cms.search.ContentSearchTool',
    
    statics:
    {
        /** 
         * @property {Number} DESCRIPTION_MAX_HEIGHT The max height for the description panel
         */
        DESCRIPTION_MAX_HEIGHT : 40,
    },
    
    /**
     * @private
     * {String} _documentationLink the url to access the documentation
     */
    _documentationLink: null,
    

    constructor: function(config)
    {        
        this.callParent(arguments);
        
        Ametys.message.MessageBus.on(Ametys.message.Message.DELETED, this._onDeleted, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.MODIFIED, this._onModified, this);
        Ametys.message.MessageBus.on(Ametys.message.Message.CREATED, this._onCreated, this);
    },
    
    setParams: function(params)
    {
        this.callParent(arguments);

        if (params.queryId)
        {
            Ametys.plugins.queriesdirectory.QueriesDAO.getQuery(params.queryId, Ext.bind(function(query){ 
                if (query)
                {
                    this._updateToolInfo(query);
                }
                else
                {
                    // Query does not exist anymore
                    var toolParams = this.getParams();
                    delete toolParams.queryId;
                    Ametys.tool.Tool.prototype.setParams.apply(this, [toolParams]); // avoid tool refreshing
                }
            }, this))
        }
    },
    
    /**
     * @private
     * Update the tool info based on the opened query
     */
    _updateToolInfo: function(query)
    {
        this.setAdditionalTitle(query.getTitle());
        
        var hasDetails = false;
        
        // Do not try to instantiate the panel in the createPanel method to avoid
        // issue with override
        var detailsPanel = this._getOrCreateDetailsPanel()
        
        var descPanel = detailsPanel.down("[itemId='description']");
        var description = query.getDescription() || "";
        descPanel.setHtml(description.replace(/\n/g, "<br/>"));
        hasDetails = description != '';
        descPanel.setVisible(hasDetails);
        
        var btn = detailsPanel.down("button[itemId='documentation-link']");
        this._documentationLink = query.getDocumentation();
        
        if (this._documentationLink)
        {
            btn.show();
            hasDetails = true;
        }
        else
        {
            btn.hide();
        }
        
        detailsPanel.setVisible(hasDetails);
    },
    
    /*
     * @private
     * Get the existing details panel or create and dock it.
     */
    _getOrCreateDetailsPanel: function()
    {
        var detailsPanel = this.mainPanel.down("[itemId='details']");
        
        if (detailsPanel != null)
        {
            return detailsPanel;
        }
        
        detailsPanel = this.mainPanel.addDocked({
            dock: 'top',
            xtype: 'container',
            cls: 'search-tool-details-panel',
            itemId: 'details',
            border: false,
            layout: {
                type: 'hbox',
                align: 'begin'
            },
            hidden: true,
            items: [
                // Collapse/expand button
                {
                    xtype: 'button',
                    iconCls: 'x-tool-collapse-top',
                    border: false,
                    itemId: 'collapse-btn',
                    tooltip: "{{i18n UITOOL_SEARCH_COLLAPSE_DETAILS_TOOLTIP}}",
                    enableToggle: true,
                    toggleHandler: this.collapseDetails,
                    scope: this,
                    cls: 'collapse-description-toggle'
                },
                // query's description
                {
                    xtype: 'container',
                    html: "",
                    itemId: 'description',
                    flex: 1,
                    hidden: true,
                    listeners: {
                        resize: this._onDescriptionResize,
                        scope: this
                    }
                },
                // query's documentation link
                {
                    xtype: 'button',
                    hidden: true, 
                    iconCls: 'ametysicon-sign-question',
                    border: false,
                    itemId: 'documentation-link',
                    text: "{{i18n UITOOL_SEARCH_OPEN_DOCUMENTATION}}",
                    tooltip: "{{i18n UITOOL_SEARCH_OPEN_DOCUMENTATION_TOOLTIP}}",
                    handler: this._openDocumentation,
                    scope: this,
                    cls: 'documentation-link'
                }
            ]
        })[0]; // retrieve the first and only one.
        
        return detailsPanel;
    },
    
    /**
     * @private
     * Toggle the display of the collapse/expand button based on the description panel size
     */
    _onDescriptionResize: function(cmp, width, height, oldWidth, oldHeight, eOpts)
    {
        var btn = this.mainPanel.down("button[itemId='collapse-btn']");
        if (height > Ametys.plugins.cms.search.ContentSearchTool.DESCRIPTION_MAX_HEIGHT)
        {
            btn.setVisible(true);
        }
        else
        {
            btn.setVisible(false);
        }
    },
    
    /**
     * Button handler to collapse or expand the details panel
     */
    collapseDetails: function(btn, state)
    {
        var detailsPanel = btn.up("[itemId='details']")
        if (state)
        {
            // collapse
            detailsPanel.setMaxHeight(Ametys.plugins.cms.search.ContentSearchTool.DESCRIPTION_MAX_HEIGHT);
            btn.setTooltip("{{i18n UITOOL_SEARCH_EXPAND_DETAILS_TOOLTIP}}");
            btn.setIconCls("x-tool-expand-bottom")
        }
        else
        {
            // expand
            detailsPanel.setMaxHeight(null);
            btn.setTooltip("{{i18n UITOOL_SEARCH_COLLAPSE_DETAILS_TOOLTIP}}");
            btn.setIconCls("x-tool-collapse-top")
        }
    },
    
    /**
     * @private
     * Open the documentation link
     */
    _openDocumentation: function()
    {
        if (this._documentationLink)
        {
            window.open(this._documentationLink, "_blank");
        }
    },
    /**
     * Listener on {@link Ametys.message.Message#DELETED} message. If the current query is concerned, the tool will be closed.
     * @param {Ametys.message.Message}  message The deleted message.
     * @protected
     */
    _onDeleted: function (message)
    {
        var queriesTargets = message.getTargets(function (target) {return Ametys.message.MessageTarget.QUERY == target.getId();});
        
        for (var i=0; i < queriesTargets.length; i++)
        {
            if (this.getParams().queryId == queriesTargets[i].getParameters().id)
            {
                this.close();
            }
            
            if (this.getParams().formattingQueryId == queriesTargets[i].getParameters().id)
            {
                var toolParams = this.getParams();
                delete toolParams.formattingQueryId;
                Ametys.tool.Tool.prototype.setParams.apply(this, [toolParams]); // avoid tool refreshing
            }
        }
    },

    /**
     * Listener on {@link Ametys.message.Message#MODIFIED} message.
     * If the current query is concerned, the tool title and details will be refreshed.
     * @param {Ametys.message.Message}  message The modified message.
     * @protected
     */
    _onModified: function (message)
    {
        var queriesTargets = message.getTargets(function (target) {return Ametys.message.MessageTarget.QUERY == target.getId();});
        
        for (var i=0; i < queriesTargets.length; i++)
        {
            if (this.getParams().queryId == queriesTargets[i].getParameters().id)
            {
                this._updateToolInfo(queriesTargets[i].getParameters().query)
            }
        }
    },
    
    /**
     * Listener on {@link Ametys.message.Message#CREATED} message.
     * If the current query is concerned, the tool title and details will be refreshed.
     * @param {Ametys.message.Message}  message The created message.
     * @protected
     */
    _onCreated: function (message)
    {
        var queriesTargets = message.getTargets(function (target) {return Ametys.message.MessageTarget.QUERY == target.getId();});
        
        for (var i=0; i < queriesTargets.length; i++)
        {
            if (this.getParams().queryId == queriesTargets[i].getParameters().id)
            {
                this._updateToolInfo(queriesTargets[i].getParameters().query);
            }
        }
    }

});
