/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class for common actions on a skin
 */
Ext.define('Ametys.plugins.skincommons.CommonSkinActions',
{
    singleton: true,

    /**
     * Commits changes.
	 * @param {String} serverRole The server role
     * @param {String} skinName The skin's name
     * @param {Boolean} quit true to quit edition after commit.
     * @param {Function} callback the callback function. Parameters are:
     * @param {Boolean} callback.success true if the operation succeeded
     * @param {Boolean} callback.quit true to quit edition
     */
    commitChanges: function (serverRole, skinName, quit, callback)
    {
        Ametys.data.ServerComm.callMethod({
            role: serverRole,
            methodName: "commitChanges",
            parameters: [skinName, quit],
            callback: {
                handler: this._commitChangesCb,
                scope: this,
                arguments: {
                    quit: quit,
                    callback: callback
                }
            },
            errorMessage:
            {
                msg: "{{i18n PLUGINS_SKINCOMMONS_COMMITCHANGES_ERROR}}",
                category: "Ametys.plugins.skincommons.actions.CommitChanges"
            },
            waitMessage: true
        });
    },

    /**
     * @private
     * Callback function invoked after commiting changes
     * @param {Object} result The result data
     * @param {Object} args The callback arguments
     */
    _commitChangesCb: function (result, args)
    {
        if (Ametys.plugins.skincommons.helper.SkinHelper.handleLockError(result))
        {
        	// The skin is currently locked
            if (Ext.isFunction(args.callback))
            {
            	args.callback(false, args.quit);
            }
            return;
        }

        var fileName = result.backupFilename;
        var adminEmail = result.adminEmail;

        var msg = "{{i18n PLUGINS_SKINCOMMONS_COMMITCHANGES_SUCCESS1}}";
        msg += '<b>' + fileName + '</b>';
        msg += "{{i18n PLUGINS_SKINCOMMONS_COMMITCHANGES_SUCCESS2}}";
        if (adminEmail != '')
        {
            msg += '(<a href="mailto:' + adminEmail + '">' + adminEmail + '</a>) ';
        }
        msg += "{{i18n PLUGINS_SKINCOMMONS_COMMITCHANGES_SUCCESS3}}";

        Ametys.Msg.show({
            title: "{{i18n PLUGINS_SKINCOMMONS_COMMITCHANGES_TITLE}}",
            msg: msg,
            buttons: Ext.Msg.OK,
            icon: Ext.MessageBox.INFO
        });

        if (Ext.isFunction(args.callback))
        {
        	args.callback(true, args.quit);
        }
    },

    /**
     * Save changes.
	 * @param {String} serverRole The server role
     * @param {String} skinName The skin's name
     * @param {Boolean} quit true to quit edition after commit.
     * @param {Function} callback the callback function. Parameters are:
     * @param {Boolean} callback.success true if the operation succeeded
     * @param {Boolean} callback.quit true to quit edition
     */
    saveChanges: function (serverRole, skinName, quit, callback)
    {
        Ametys.data.ServerComm.callMethod({
            role: serverRole,
            methodName: "saveChanges",
            parameters: [skinName, quit],
            callback: {
                handler: this._saveChangesCb,
                scope: this,
                arguments: {
                    quit: quit,
                    callback: callback
                }
            },
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINCOMMONS_COMMITCHANGES_ERROR}}",
                category: this.self.getName()
            },
            waitMessage: true
        });
    },

    /**
     * Callback invoked after saving changes
     * @param {Object} response The response from the server
     * @param {Object} args The callback arguments
     * @param {Boolean} args.quit true to quit edition after saving
     * @param {Function} args.callback callback function
     * @private
     */
    _saveChangesCb: function (response, args)
    {
    	if (Ametys.plugins.skincommons.helper.SkinHelper.handleLockError(response))
        {
            if (Ext.isFunction(args.callback))
            {
            	args.callback(false, args.quit);
            }
            return;
        }
    	
        if (Ext.isFunction(args.callback))
        {
        	args.callback(true, args.quit);
        }
    },
    
    /**
     * Clear all modifications
	 * @param {String} serverRole The server role
     * @param {String} skinName The skin's name
     * @param {Boolean} workVersion true to get back the work version
     * @param {Function} callback the callback function. Parameters are:
     * @param {Boolean} callback.success true if the operation succeeded
     */
    clearModifications: function (serverRole, skinName, workVersion, callback)
    {
        Ametys.data.ServerComm.callMethod({
            role: serverRole,
            methodName: "clearModifications",
            parameters: [skinName, workVersion],
            callback: {
                handler: this._clearModificationsCb,
                scope: this,
                arguments: {
                    callback: callback
                }
            },
            errorMessage:
            {
                msg: "{{i18n PLUGINS_SKINCOMMONS_CLEAR_MODIFICATION_ERROR}}",
                category: this.self.getName()
            },
            waitMessage: true
        });
    },
    
    /**
     * Callback invoked after clearing changes
     * @param {Object} response The response from the server
     * @param {Object} args The callback arguments
     * @param {Function} args.callback callback function
     * @private
     */
    _clearModificationsCb: function (response, args)
    {
    	if (Ametys.plugins.skincommons.helper.SkinHelper.handleLockError(response))
        {
            if (Ext.isFunction(args.callback))
            {
            	args.callback(false);
            }
            return;
        }
    	
        if (Ext.isFunction(args.callback))
        {
        	args.callback(true);
        }
    },

    /**
     * Cancel changes.
	 * @param {String} serverRole The server role
     * @param {String} skinName The skin's name
     * @param {Boolean} workVersion true to get back the work version
     * @param {String} toolId The tool's id.
     * @param {Function} callback the callback function. Parameters are:
     * @param {Boolean} callback.success true if the operation succeeded
     */
    cancelChanges: function (serverRole, skinName, workVersion, toolId, callback)
    {
    	Ametys.data.ServerComm.callMethod({
            role: serverRole,
            methodName: "cancelChanges",
            parameters: [skinName, workVersion, toolId],
            callback: {
                handler: this._cancelChangesCb,
                arguments: {
                    callback: callback
                },
                scope: this
            },
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINCOMMONS_CANCELCHANGES_ERROR}}",
                category: this.self.getName()
            },
            waitMessage: true
        });
    },
    
    /**
     * Callback invoked after cancelling changes
     * @param {Object} response The response from the server
     * @param {Object} args The callback arguments
     * @param {Function} args.callback callback function.
     * @private
     */
    _cancelChangesCb: function (response, args)
    {
    	if (Ametys.plugins.skincommons.helper.SkinHelper.handleLockError(response))
        {
            if (Ext.isFunction(args.callback))
            {
            	args.callback(false, response.hasChanges);
            }
            return;
        }
    	
    	if (Ext.isFunction(args.callback))
        {
        	args.callback(true, response.hasChanges);
        }
    }
    
});
