/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.skincommons;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;

/**
 * Used to protect skin edition pipelines.
 * Stops the pipelines execution if the skin is locked by another user.
 * If locked, set a JSON map in request attributes : {isLocked: true, lockOwner: XXX}
 */
public class CheckNotLockedAction extends ServiceableAction
{
    private UserManager _userManager;
    private SkinLockManager _lockManager;
    private SkinEditionHelper _skinHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _lockManager = (SkinLockManager) smanager.lookup(SkinLockManager.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _skinHelper = (SkinEditionHelper) smanager.lookup(SkinEditionHelper.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String skinName = request.getParameter("skinName");
        
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        
        if (!_lockManager.canWrite(tempDir))
        {
            Map<String, Object> result = new HashMap<>();
            
            UserIdentity lockOwner = _lockManager.getLockOwner(tempDir);
            User user = _userManager.getUser(lockOwner.getPopulationId(), lockOwner.getLogin());

            result.put("isLocked", true);
            result.put("lockOwner", user != null ? user.getFullName() + " (" + lockOwner + ")" : lockOwner);
            result.put("success", false);
            
            request.setAttribute(JSonReader.OBJECT_TO_READ, result);
            
            // Stop pipelines execution
            return EMPTY_MAP;
        }
        
        // Continue
        return null;
    }

}
