/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.skineditor.readers;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.reading.ServiceableReader;
import org.apache.commons.io.IOUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.impl.FileSource;
import org.xml.sax.SAXException;

import org.ametys.core.util.ImageHelper;
import org.ametys.core.util.URIUtils;

/**
 * Reader for temporary skin resources
 */
public class SkinResourceReader extends ServiceableReader
{
    /** The source resolver */
    protected org.apache.excalibur.source.SourceResolver _srcResolver;
    /** The file source */
    protected FileSource _file;
    
    private boolean _readForDownload;
    
    private int _width;
    private int _height;
    private int _maxWidth;
    private int _maxHeight;
    private Collection<String> _allowedFormats = Arrays.asList(new String[]{"png", "gif", "jpg", "jpeg"});
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _srcResolver = (org.apache.excalibur.source.SourceResolver) smanager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
    }
    
    @Override
    public void setup(SourceResolver res, Map objModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        super.setup(res, objModel, src, par);
        
        String skin = par.getParameter("skin", null);
        String path = par.getParameter("path", null);
        
        assert skin != null || path != null;
        
        _readForDownload = par.getParameterAsBoolean("download", false);
        
        // parameters for image resizing
        _width = par.getParameterAsInteger("width", 0);
        _height = par.getParameterAsInteger("height", 0);
        _maxWidth = par.getParameterAsInteger("maxWidth", 0);
        _maxHeight = par.getParameterAsInteger("maxHeight", 0);
        
        _file = (FileSource) _getSource(skin, path);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        try (InputStream is = _file.getInputStream();)
        {
            String name = _file.getName();
            name = name.replaceAll("\\\\", "\\\\\\\\");
            name = name.replaceAll("\\\"", "\\\\\\\"");
            
            Response response = ObjectModelHelper.getResponse(objectModel);
            
            if (_readForDownload)
            {
                response.setHeader("Content-Disposition", "attachment; filename=\"" + name + "\"");
            }
            
            if (_width > 0 || _height > 0 || _maxHeight > 0 || _maxWidth > 0)
            {
                // it's an image, which must be resized
                int i = name.lastIndexOf('.');
                String format = i != -1 ? name.substring(i + 1) : "png";
                format = _allowedFormats.contains(format) ? format : "png";
                
                ImageHelper.generateThumbnail(is, out, format, _height, _width, _maxHeight, _maxWidth);
            }
            else
            {
                response.setHeader("Content-Length", Long.toString(_file.getContentLength()));

                IOUtils.copy(is, out);
            }
        }
        finally 
        {
            IOUtils.closeQuietly(out);
        }
    }
    
    private Source _getSource (String skinName, String path) throws IOException
    { 
        String decodedPath = URIUtils.decode(path);
        decodedPath = decodedPath.startsWith("/")  ? decodedPath : "/" + decodedPath;
        
        return _srcResolver.resolveURI("ametys-home://skins/temp/" + skinName + decodedPath);
    }
}
