/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.skineditor.resources;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.FileSource;

import org.ametys.core.file.FileHelper;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.skincommons.SkinEditionHelper;
import org.ametys.plugins.skincommons.SkinLockManager;
import org.ametys.plugins.skineditor.skin.SkinDAO;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinsManager;

/**
 * DAO for files and folders inside a skin directory
 */
public class SkinResourceDAO implements Serviceable, Component
{
    /** Constant for skin editor tool id */
    public static final String ROLE = SkinResourceDAO.class.getName();
    /** Constant for skin editor tool id */
    public static final String SKIN_EDITOR_TOOL_ID = "uitool-skineditor";
    
    /** The lock manager */
    protected SkinLockManager _lockManager;
    /** The skin edition helper */
    protected SkinEditionHelper _skinHelper;
    /** The file helper */
    protected FileHelper _fileHelper;
    /** The source resolver */
    protected SourceResolver _srcResolver;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The rights manager */
    protected RightManager _rightManager;
    /** The skins manager */
    protected SkinsManager _skinManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _skinHelper = (SkinEditionHelper) manager.lookup(SkinEditionHelper.ROLE);
        _skinManager = (SkinsManager) manager.lookup(SkinsManager.ROLE);
        _fileHelper = (FileHelper) manager.lookup(FileHelper.ROLE);
        _lockManager = (SkinLockManager) manager.lookup(SkinLockManager.ROLE);
        _srcResolver = (org.apache.excalibur.source.SourceResolver) manager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    /**
     * Saves the text in file
     * @param skinName The name of the skin containing the resource
     * @param relPath the relative path of file under the skin directory
     * @param text the file content to save
     * @return The result map
     * @throws IOException If an error occurred while saving
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> save(String skinName, String relPath, String text) throws IOException
    {
        _checkUserRight(skinName);
        
        String tempDirURI = _skinHelper.getTempDirectoryURI(skinName);
        
        String fileURI = tempDirURI + (relPath.length() > 0 ? "/" + relPath : "");
        
        Map<String, Object> result = _fileHelper.saveFile(fileURI, text);
        
        if (result.containsKey("isI18n"))
        {
            _skinHelper.invalidateTempSkinCatalogues(skinName);
        }
        
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        // Update lock
        _lockManager.updateLockFile(tempDir, SKIN_EDITOR_TOOL_ID);
        
        return result;
    }
    
    /**
     * checks if the resource already exists in the parent
     * @param skinName The name of the skin containing the resource
     * @param parentRelPath The parent path
     * @param fileName the file name to check
     * @return true if the file exists
     * @throws IOException if something goes wrong while trying to retrieve a file
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public boolean checkSourceExists(String skinName, String parentRelPath, String fileName) throws IOException
    {
        _checkUserRight(skinName);
        
        String tempDirURI = _skinHelper.getTempDirectoryURI(skinName);
        String parentFileURI = tempDirURI + (parentRelPath.length() > 0 ? "/" + parentRelPath : "");
        
        return _fileHelper.hasChild(parentFileURI, fileName);
    }
    
    /**
     * Copy a file or directory
     * @param skinName the name of the current skin
     * @param srcPath the path of the source file or directory
     * @param parentTargetPath the new path for the source file or directory
     * @return a map of data
     * @throws IOException if something went wrong during the source copy processing
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> copySource(String skinName, String srcPath, String parentTargetPath) throws IOException
    {
        _checkUserRight(skinName);
        
        String tempDirURI = _skinHelper.getTempDirectoryURI(skinName);
        
        String srcFileURI = tempDirURI + (srcPath.length() > 0 ? "/" + srcPath : "");
        String parentTargetURI = tempDirURI + (parentTargetPath.length() > 0 ? "/" + parentTargetPath : "");
        
        Map<String, Object> result = _fileHelper.copySource(srcFileURI, parentTargetURI);
        
        // Update lock
        FileSource rootDir = (FileSource) _srcResolver.resolveURI(tempDirURI);
        _lockManager.updateLockFile(rootDir.getFile().toPath(), SKIN_EDITOR_TOOL_ID);
        
        if (result.containsKey("uri"))
        {
            String folderUri = (String) result.get("uri");
            String path = folderUri.substring(rootDir.getURI().length());
            result.put("path", path.endsWith("/") ? path.substring(0, path.length() - 1) : path);
            result.put("skinName", skinName);
        }
        
        return result;
    }
    
    /**
     * Create a folder
     * @param skinName the name of the current skin
     * @param parentRelPath the path of the parent containing the folder
     * @param originalName the name of the new folder
     * @param renameIfExists if true, will generate a valid name if "originalName" already exists.
     * @return a map of data
     * @throws IOException if an error occurs while manipulating files
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map addFolder(String skinName, String parentRelPath, String originalName, boolean renameIfExists) throws IOException
    {
        _checkUserRight(skinName);
        
        String tempDirURI = _skinHelper.getTempDirectoryURI(skinName);
        String parentURI = tempDirURI + (parentRelPath.length() > 0 ? "/" + parentRelPath : "");
        
        Map<String, Object> result = _fileHelper.addFolder(parentURI, originalName, renameIfExists);
        
        // Update lock
        FileSource rootDir = (FileSource) _srcResolver.resolveURI(tempDirURI);
        _lockManager.updateLockFile(rootDir.getFile().toPath(), SKIN_EDITOR_TOOL_ID);
       
        if (result.containsKey("uri"))
        {
            String folderUri = (String) result.get("uri");
            String path = folderUri.substring(rootDir.getURI().length());
            result.put("path", path.endsWith("/") ? path.substring(0, path.length() - 1) : path);
            result.put("parentPath", parentRelPath.endsWith("/") ? parentRelPath.substring(0, parentRelPath.length() - 1) : parentRelPath);
            result.put("skinName", skinName);
        }
        
        return result;
    }
    
    /**
     * Add or update a file into a skin
     * @param skinName the name of the skin
     * @param part The file multipart to upload
     * @param toPath The path of parent directory
     * @param mode The insertion mode: 'add-rename' or 'update' or null.
     * @param unzip true to unzip .zip file
     * @return the result map
     * @throws IOException If an error occurred manipulating the file
     */
    public Map addOrUpdateFile(String skinName, Part part, String toPath, String mode, boolean unzip) throws IOException
    {
        _checkUserRight(skinName);
        
        String tempDirURI = _skinHelper.getTempDirectoryURI(skinName);
        FileSource tempDir = (FileSource) _srcResolver.resolveURI(tempDirURI);
        
        String parentURI = tempDirURI + (toPath.length() > 0 ? "/" + toPath : "");
        FileSource parentDir = (FileSource) _srcResolver.resolveURI(parentURI);
        
        Map<String, Object> result = _fileHelper.addOrUpdateFile(part, parentDir, mode, unzip);
        
        // Update lock
        _lockManager.updateLockFile(tempDir.getFile().toPath(), SKIN_EDITOR_TOOL_ID);
        
        if (result.containsKey("uri"))
        {
            String fileUri = (String) result.get("uri");
            String filePath = fileUri.substring(tempDir.getURI().length());
            result.put("path", filePath.endsWith("/") ? filePath.substring(0, filePath.length() - 1) : filePath);
            String parentPath = parentDir.getURI().substring(tempDir.getURI().length());
            result.put("parentPath", parentPath.endsWith("/") ? parentPath.substring(0, parentPath.length() - 1) : parentPath);
        }
        else if (result.containsKey("unzip"))
        {
            String parentPath = parentDir.getURI().substring(parentDir.getURI().length());
            result.put("parentPath", parentPath.endsWith("/") ? parentPath.substring(0, parentPath.length() - 1) : parentPath);
        }
        
        return result;
    }


    /**
     * Delete a file or a directory
     * @param skinName the name of the current skin
     * @param relPath the path of the file or directory
     * @return a map of data
     * @throws IOException if an error occurs while manipulating files
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map deleteFile(String skinName, String relPath) throws IOException
    {
        _checkUserRight(skinName);
        
        String tempDirURI = _skinHelper.getTempDirectoryURI(skinName);
        String fileURI = tempDirURI + (relPath.length() > 0 ? "/" + relPath : "");
        
        Map result = _fileHelper.deleteFile(fileURI);
        
        // Update lock
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        _lockManager.updateLockFile(tempDir, SKIN_EDITOR_TOOL_ID);
        
        return result;
    }
    
    /**
     * Move a file or a directory
     * @param skinName the name of current skin
     * @param srcPath the path of the file or directory
     * @param targetPath the targeted path
     * @return a map of data
     * @throws IOException if something goes wrong during the source moving process
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> moveSource(String skinName, String srcPath, String targetPath) throws IOException
    {
        _checkUserRight(skinName);
        
        String tempDirURI = _skinHelper.getTempDirectoryURI(skinName);
        
        String srcFileURI = tempDirURI + (srcPath.length() > 0 ? "/" + srcPath : "");
        String parentTargetURI = tempDirURI + (targetPath.length() > 0 ? "/" + targetPath : "");
        
        Map<String, Object> result = _fileHelper.moveSource(srcFileURI, parentTargetURI);
        
        // Update lock
        FileSource rootDir = (FileSource) _srcResolver.resolveURI(tempDirURI);
        _lockManager.updateLockFile(rootDir.getFile().toPath(), SKIN_EDITOR_TOOL_ID);

        if (result.containsKey("uri"))
        {
            String folderUri = (String) result.get("uri");
            String path = folderUri.substring(rootDir.getURI().length());
            result.put("path", path.endsWith("/") ? path.substring(0, path.length() - 1) : path);
            result.put("skinName", skinName);
        }
        
        return result;
    }
    
    /**
     * Rename a file or a directory
     * @param skinName the current skin name
     * @param relPath the path of the file
     * @param name the new name
     * @return a map of data
     * @throws IOException if something goes wrong when renaming the source
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map renameSource(String skinName, String relPath, String name) throws IOException
    {
        _checkUserRight(skinName);
        
        String tempDirURI = _skinHelper.getTempDirectoryURI(skinName);
        String fileURI = tempDirURI + (relPath.length() > 0 ? "/" + relPath : "");
        
        Map<String, Object> result = _fileHelper.renameFile(fileURI, name);
        
        // Update lock
        FileSource rootDir = null;
        try
        {
            rootDir = (FileSource) _srcResolver.resolveURI(tempDirURI);
            _lockManager.updateLockFile(rootDir.getFile().toPath(), SKIN_EDITOR_TOOL_ID);
        }
        finally
        {
            _srcResolver.release(rootDir);
        }
        
        if (result.containsKey("uri"))
        {
            String newURI = (String) result.get("uri");
            String path = newURI.substring(rootDir.getURI().length());
            result.put("path", path);
            result.put("name", name);
            result.put("skinName", skinName);
        }
        
        return result;
    }
    
    private void _checkUserRight(String skinName) throws IllegalStateException
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        Skin skin = _skinManager.getSkin(skinName);
        if (skin == null)
        {
            throw new IllegalStateException("User '" + user + "' tried to modify an unknown skin '" + skinName + "'");
        }
        
        if (!skin.isModifiable())
        {
            throw new IllegalStateException("User '" + user + "' tried to modify an unmodifiable skin '" + skinName + "'");
        }
        
        if (!(_rightManager.hasRight(user, SkinDAO.EDIT_SKINS_RIGHT_ID, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW || _rightManager.hasRight(user, SkinDAO.EDIT_CURRENT_SKIN_RIGHT_ID, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW && skinName.equals(_skinManager.getSkinNameFromRequest())))
        {
            throw new IllegalStateException("User '" + user + "' tried to modify skin '" + skinName + "' without convenient right.");
        }
    }
}
