/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This UI helper provides a dialog box to choose a skin
 * See #open method.
 * @private
 */
Ext.define('Ametys.plugins.skineditor.skin.ChooseSkin',
{
    singleton: true,

    /**
     * @property {Boolean} _initialized True if the dialog box has been initialized.
     * @private
     */
    /**
     * @property {Ext.form.Basic} _form The inner basic form.
     * @private
     */
    /**
     * @property {Ametys.window.DialogBox} _box The dialog box
     * @private
     */


    /**
     * Allow the user to upload a file from local hard drive
     * @param {Object[]} skinsList The list of skins to choose from
     * @param {Function} callback The callback function called when the file was uploaded. Has the following parameters
     * @param {String} callback.path The path of uploaded file
     * @param {String} callback.parentPath The path of parent folder
     * @param {Boolean} callback.reload True if a reload is needed
     */
    open: function (skinsList, callback)
    {
        this._cbFn = callback;

        this._delayedInitialize();
        this._box.show();
        this._initCombo(skinsList);
    },

    /**
     * Initialize the dialog box
     * @private
     */
    _delayedInitialize: function ()
    {
        if (this._initialized)
        {
            return;
        }

        this._skinCombo = Ext.create('Ext.form.field.ComboBox', {
            fieldLabel: "{{i18n PLUGINS_SKINEDITOR_SKINNAME}}",
            name: 'skin',
            itemId: 'skin',
            
            queryMode: 'local',
            editable: false,
            forceSelection: true,
            triggerAction: 'all',
            store: Ext.create('Ext.data.Store',
            {
                fields: [ 'id', 'label' ]
            }),

            valueField: 'id',
            displayField: 'label',

            emptyText: "{{i18n PLUGINS_SKINEDITOR_SELECT_SKIN}}",
            allowBlank: false
        });

        var formPanel = Ext.create('Ext.form.Panel',
        {
            border: false,
            scrollable: true,
            defaults:
            {
                cls: 'ametys',
                anchor: '95%',
                labelAlign: 'right',
                labelSeparator: '',
                labelWidth: 55
            },

            items: [
                {
                    xtype: 'container',
                    html: "{{i18n PLUGINS_SKINEDITOR_SELECT_SKIN_HINT_1}}"
                },
                this._skinCombo,
                {
                    xtype: 'container',
                    html: "{{i18n PLUGINS_SKINEDITOR_SELECT_SKIN_HINT_2}}"
                }
            ]
        });

        this._box = Ext.create('Ametys.window.DialogBox',
        {
            title: "{{i18n PLUGINS_SKINEDITOR_TOOL_LABEL}}",
            iconCls: 'ametysicon-html25 decorator-ametysicon-painter14',

            layout: 'fit',
            width: 430,

            items: formPanel,

            defaultFocus: 'skin',
            closeAction: 'hide',

            referenceHolder: true,
            defaultButton: 'validate',
            
            buttons: [{
            	reference: 'validate',
                text: "{{i18n PLUGINS_SKINEDITOR_DIALOG_OK}}",
                handler: Ext.bind(this._ok, this)
            }, {
                text: "{{i18n PLUGINS_SKINEDITOR_DIALOG_CANCEL}}",
                handler: Ext.bind(this._cancel, this)
            }]
        });

        this._initialized = true;
    },

    /**
     * Initialize or reinitialize the form with the appropriate values.
     * @param {String} skins The list of skins
     * @private
     */
    _initCombo: function (skins)
    {
        var store = this._skinCombo.getStore();
        store.removeAll();

        for (var i = 0; i < skins.length; i++)
        {
            store.add({
                'id' : skins[i].id,
                'label' : skins[i].label + ' (' + skins[i].id + ')'
            });

            if (skins[i].current == true)
            {
                this._skinCombo.setValue(skins[i].id);
            }
        }

        this._skinCombo.clearInvalid();
    },

    /**
     * This function is called when the 'Ok' button of the dialog box is pressed
     * send the selected value to the callback
     * @private
     */
    _ok: function ()
    {
        if (this._skinCombo.isValid())
        {
            var skin = this._skinCombo.getValue();

            if (Ext.isFunction(this._cbFn))
            {
                this._cbFn(skin);
            }

            this._box.hide();
        }
    },

    /**
     * This function is called when the 'cancel' button of the dialog box is pressed
     * @private
     */
    _cancel: function ()
    {
        if (Ext.isFunction(this._cbFn))
        {
            this._cbFn(null);
        }

        this._box.hide();
    }

});
