/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.skineditor.clientsideelement;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.skineditor.skin.SkinDAO;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinsManager;

/**
 * Client side element to open the skin editor. It is available if at least one skin is modifiable
 *
 */
public class OpenSkinEditorClientSideElement extends StaticClientSideElement
{
    private SkinsManager _skinManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _skinManager = (SkinsManager) smanager.lookup(SkinsManager.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        if (!_hasModifiableSkin())
        {
            return Collections.EMPTY_LIST;
        }
        else
        {
            return super.getScripts(ignoreRights, contextParameters);
        }
    }
    
    private boolean _hasModifiableSkin()
    {
        UserIdentity user = _currentUserProvider.getUser();
        if (_rightManager.hasRight(user, SkinDAO.EDIT_SKINS_RIGHT_ID, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW)
        {
            return !_skinManager.getSkins().stream()
                    .map(id -> _skinManager.getSkin(id))
                    .filter(Skin::isModifiable)
                    .collect(Collectors.toList()).isEmpty();
        }
        else if (_rightManager.hasRight(user, SkinDAO.EDIT_CURRENT_SKIN_RIGHT_ID, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW)
        {
            String currentSkinName = _skinManager.getSkinNameFromRequest();
            return _skinManager.getSkin(currentSkinName).isModifiable();
        }
        return false;
    }
}
