/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Menu for the variants gallery
 */
Ext.define('Ametys.plugins.skinfactory.widgets.gallery.VariantsGallery', {
	extend: 'Ext.menu.Menu',
	
	constructor: function (config) 
	{
		this.callParent(arguments);
		
		this.on("show", this._onMenuShow, this);
		this.on('afterrender', this._onAfterRender, this);
		
		Ametys.message.MessageBus.on(Ametys.message.Message.LOADED, this._onSkinLoaded, this);
	},
	
	/**
	 * @private
	 * Listener invoked after rendering item.
	 * Apply the parent menu UI to the item
	 */
	_onAfterRender: function()
	{
		if (this.parentMenu && this.parentMenu.ui)
		{
			this.setUI(this.parentMenu.ui);
		}
	},
	
    /**
     * @private
     * Listener for message bus that says the skin is loaded
     * @param {Ametys.message.Message} message The loaded message
     */
	_onSkinLoaded: function (message)
	{
		var target = message.getTarget(Ametys.message.MessageTarget.SKIN);
        if (target)
        {
            // Remove old gallery items
            this.removeAll();
            
        	// Load the gallery items
    		var tool = Ametys.tool.ToolsManager.getTool("uitool-skinfactory");
            var skinName = tool.getSkinName();
    		Ametys.plugins.skinfactory.skin.SkinDAO.getGalleryVariants([skinName, this.paramId], Ext.bind(this._loadItemsCb, this));
        }
	},
	
	/**
	 * Listener when a button is clicked
	 * @param {Ametys.ui.fluent.ribbon.controls.Button} button This button
	 */
	_onClick: function (button)
	{
		Ametys.plugins.skinfactory.skin.SkinActions.updateParameter(this.getInitialConfig("paramId"), button.value);
	},
	
	/**
	 * Listener when the menu of the button is showed.
	 * @param {Ext.menu.Menu} menu The menu.
	 */
	_onMenuShow: function (menu)
	{
		var currentValue = Ametys.plugins.skinfactory.SkinParametersManager.getParameterValue(this.getInitialConfig("paramId"));
		
		var menuPanel = menu.items.get(0);
		
		// Iterating over each item in galleries
		menuPanel.items.each(function (menuItem) {
			menuItem.toggle(currentValue != null && menuItem.value == currentValue);
		});
	},
	
	/**
	 * Callback after retrieving the list of gallery variants. Create the menu gallery
	 * @param {Object[]} galleryVariants The list of gallery variants
	 */
	_loadItemsCb: function (galleryVariants)
	{
		var items = [];
		var me = this;
		
		Ext.Array.each(galleryVariants, function (galleryVariant) {
			var item = Ext.create("Ametys.ui.fluent.ribbon.controls.Button", {
				text: galleryVariant.label,
				tooltip: galleryVariant.description,
				icon: Ametys.CONTEXT_PATH + galleryVariant.thumbnail,
				scale: 'large',
				handler: Ext.bind(me._onClick, me),
				enableToggle: true,
				value: galleryVariant.value,
				allowDepress: false
			});
			items.push(item);
		});
		
		if (items.length > 0)
		{
			var menuGallery = Ext.create("Ametys.ui.fluent.ribbon.controls.gallery.MenuPanel", {
				title: "{{i18n PLUGINS_SKINFACTORY_VARIANTGALLERY_GROUP_LABEL}}", 
				items: items,
				
				width: 402,
				defaults: {
					width: 80
				}
			});
			this.add(menuGallery);
			this.setDisabled(false);
		}
		else
		{
			this.setDisabled(true);
		}
	}
});
