/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.clientsidelement;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;

import org.ametys.core.ui.ClientSideElement;
import org.ametys.core.ui.SimpleMenu;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.util.I18nUtils;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.skinfactory.model.ModelDesignsManager;
import org.ametys.skinfactory.model.ModelDesignsManager.Design;
import org.ametys.web.skin.SkinModelsManager;

/**
 * {@link StaticClientSideElement} for design of model
 *
 */
public class DesignsMenu extends SimpleMenu
{
    private ModelDesignsManager _designsManager;
    private boolean _designsInitialized;
    private SkinModelsManager _modelsManager;
    private I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _designsManager = (ModelDesignsManager) smanager.lookup(ModelDesignsManager.ROLE);
        _modelsManager = (SkinModelsManager) smanager.lookup(SkinModelsManager.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }

    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        Set<String> models = _modelsManager.getModels();
        if (models != null)
        {
            // Test to find if at lease one model has a design (if not, no scripts are added)
            boolean hasDesigns = models.stream()
                    .map(_designsManager::getDesigns)
                    .anyMatch(d -> !d.isEmpty());
            
            if (hasDesigns)
            {
                return super.getScripts(ignoreRights, contextParameters);
            }
        }
        
        return Collections.EMPTY_LIST;
    }
    
    @Override
    protected void _getGalleryItems(Map<String, Object> parameters, Map<String, Object> contextualParameters)
    {
        try
        {
            _lazyInitializeDesignsGallery();
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Unable to lookup client side element local components", e);
        }
        
        if (_galleryItems.size() > 0)
        {
            parameters.put("gallery-item", new LinkedHashMap<>());
            
            for (GalleryItem galleryItem : _galleryItems)
            {
                @SuppressWarnings("unchecked")
                Map<String, Object> galleryItems = (Map<String, Object>) parameters.get("gallery-item");
                galleryItems.put("gallery-groups", new ArrayList<>());
                
                for (GalleryGroup galleryGp : galleryItem.getGroups())
                {
                    @SuppressWarnings("unchecked")
                    List<Object> galleryGroups = (List<Object>) galleryItems.get("gallery-groups");
                    
                    Map<String, Object> groupParams = new LinkedHashMap<>();
                    
                    I18nizableText label = galleryGp.getLabel();
                    groupParams.put("label", label);
                    
                    // Group items
                    List<String> gpItems = new ArrayList<>();
                    for (ClientSideElement element : galleryGp.getItems())
                    {
                        gpItems.add(element.getId());
                    }
                    
                    if (gpItems.size() > 0)
                    {
                        groupParams.put("items", gpItems);
                        galleryGroups.add(groupParams);
                    }
                }
            }
        }
    }
    

    /**
     * Lazy initialization of the design gallery
     * @throws ConfigurationException if the design configuration is wrong
     * @throws ProcessingException if an error occurs when retrieving the models
     */
    private synchronized void _lazyInitializeDesignsGallery() throws ConfigurationException, ProcessingException
    {
        if (!_designsInitialized)
        {
            Set<String> models = _modelsManager.getModels();
            
            GalleryItem galleryItem = new GalleryItem();
            
            GalleryGroup galleryGroup = new GalleryGroup(new I18nizableText("plugin." + _pluginName, "PLUGINS_SKINFACTORY_DESIGNMENU_GROUP_LABEL"));
            galleryItem.addGroup(galleryGroup);

            for (String modelName : models)
            {
                Set<Design> designs = _designsManager.getDesigns(modelName);
            
                if (designs.size() > 0)
                {
                    for (Design design : designs)
                    {
                        String itemId = this.getId() + "-" + modelName + "-" + design.getId();
                        
                        Configuration conf = _getDesignConfiguration(itemId, design, modelName);
                        _getGalleryItemManager().addComponent(_pluginName, null, itemId, StaticClientSideElement.class, conf);
                        galleryGroup.addItem(new UnresolvedItem(itemId, true));
                    }
                }
            }
            
            _galleryItems.add(galleryItem);
            
            try
            {
                _getGalleryItemManager().initialize();
            }
            catch (Exception e)
            {
                throw new ConfigurationException("Unable to lookup parameter local components", e);
            }
            
            _designsInitialized = true;
        }
    }

    /**
     * Get the configuration of the model item
     * @param id The id of item
     * @param design The design
     * @param modelName The model name
     * @return The configuration
     */
    protected Configuration _getDesignConfiguration(String id, Design design, String modelName)
    {
        DefaultConfiguration conf = new DefaultConfiguration("extension");
        conf.setAttribute("id", id);
        
        DefaultConfiguration classConf = new DefaultConfiguration("class");
        classConf.setAttribute("name", "Ametys.ribbon.element.ui.ButtonController");
        
        // Label
        DefaultConfiguration labelConf = new DefaultConfiguration("label");
        labelConf.setValue(_i18nUtils.translate(design.getLabel()));
        classConf.addChild(labelConf);
        
        // Description
        DefaultConfiguration descriptionConf = new DefaultConfiguration("description");
        descriptionConf.setValue(_i18nUtils.translate(design.getDescription()));
        classConf.addChild(descriptionConf);
        
        // Icons
        DefaultConfiguration iconSmallConf = new DefaultConfiguration("icon-small");
        iconSmallConf.setValue(design.getIcon());
        classConf.addChild(iconSmallConf);
        DefaultConfiguration iconMediumConf = new DefaultConfiguration("icon-medium");
        iconMediumConf.setValue(design.getIcon());
        classConf.addChild(iconMediumConf);
        DefaultConfiguration iconLargeConf = new DefaultConfiguration("icon-large");
        iconLargeConf.setValue(design.getIcon());
        classConf.addChild(iconLargeConf);

        // Model name
        DefaultConfiguration modelNameConf = new DefaultConfiguration("modelName");
        modelNameConf.setValue(modelName);
        classConf.addChild(modelNameConf);
        
        // Design id
        DefaultConfiguration designConf = new DefaultConfiguration("designId");
        designConf.setValue(design.getId());
        classConf.addChild(designConf);
        
        // Common configuration
        @SuppressWarnings("unchecked")
        Map<String, Object> commonConfig = (Map<String, Object>) this._script.getParameters().get("items-config");
        for (String tagName : commonConfig.keySet())
        {
            DefaultConfiguration c = new DefaultConfiguration(tagName);
            c.setValue(String.valueOf(commonConfig.get(tagName)));
            classConf.addChild(c);
        }
        
        conf.addChild(classConf);
        return conf;
    }
}
