/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.skinfactory.model;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.io.FileUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.util.path.PathUtils;
import org.ametys.plugins.skincommons.SkinEditionHelper;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.util.AmetysHomeHelper;
import org.ametys.skinfactory.SkinFactoryComponent;
import org.ametys.skinfactory.filefilter.FileFilter;
import org.ametys.web.cocoon.I18nTransformer;
import org.ametys.web.cocoon.I18nUtils;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinDAO;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModelsManager;
import org.ametys.web.skin.SkinsManager;

/**
 * Component for interact with a skin model
 */
public class SkinModelDAO extends AbstractLogEnabled implements Serviceable, Component
{
    private static final DateFormat _DATE_FORMAT = new SimpleDateFormat("yyyyMMdd-HHmm");
    
    
    private SkinsManager _skinsManager;
    private SkinModelsManager _modelsManager;
    private SkinFactoryComponent _skinFactoryManager;
    private SkinEditionHelper _skinHelper;
    private SkinDAO _skinDAO;
    private I18nUtils _i18nUtils;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _skinsManager = (SkinsManager) manager.lookup(SkinsManager.ROLE);
        _modelsManager = (SkinModelsManager) manager.lookup(SkinModelsManager.ROLE);
        _skinFactoryManager = (SkinFactoryComponent) manager.lookup(SkinFactoryComponent.ROLE);
        _skinHelper = (SkinEditionHelper) manager.lookup(SkinEditionHelper.ROLE);
        _skinDAO = (SkinDAO) manager.lookup(SkinDAO.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(org.ametys.core.util.I18nUtils.ROLE);
    }

    /**
     * Retrieve informations on a skin
     * @param modelId The skin id
     * @return the informations of a skin
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public Map<String, Object> getModel(String modelId)
    {
        Map<String, Object> result = new HashMap<>();

        SkinModel model = _modelsManager.getModel(modelId);

        if (model != null)
        {
            result.put("name", model.getId());
            result.put("title", model.getLabel());
            result.put("isModifiable", model.isModifiable());
        }
        
        return result;
    }
    
    /**
     * Retrieve the list of models and skins available
     * @param includeAbstract Should include abstract skins
     * @return a map of skins and models
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public Map<String, Object> getSkinsAndModels(boolean includeAbstract)
    {
        Map<String, Object> result = _skinDAO.getSkins(includeAbstract);

        result.put("models", _models2JsonObject());

        return result;
    }

    private List<Object> _models2JsonObject()
    {
        List<Object> modelsList = new ArrayList<>();
        Set<String> models = _modelsManager.getModels();
        for (String modelName : models)
        {
            Map<String, Object> jsonModel = new HashMap<>();
            SkinModel model = _modelsManager.getModel(modelName);

            jsonModel.put("id", modelName);
            jsonModel.put("label", model.getLabel());
            jsonModel.put("description", model.getDescription());
            jsonModel.put("iconLarge", model.getLargeImage());
            jsonModel.put("iconSmall", model.getSmallImage());

            modelsList.add(jsonModel);
        }

        return modelsList;
    }
    
    /**
     * Determines if a model exists
     * @param modelId The model id
     * @return true if model exists.
     * @throws ProcessingException if something goes wrong when retrieving the list of models
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public boolean modelExists (String modelId) throws ProcessingException
    {
        return _modelsManager.getModels().contains(modelId);
    }
    
    /**
     * Import a model from a zip file
     * @param modelName The name of the new model
     * @param tmpDirPath the tmp dir path where the zip has been uploaded
     * @return The model name
     * @throws IOException if something goes wrong when manipulating files
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public String importModel(String modelName, String tmpDirPath) throws IOException
    {
        Path tmpDir = AmetysHomeHelper.getAmetysHomeTmp().toPath().resolve(tmpDirPath);
        
        if (Files.isDirectory(tmpDir))
        {
            // If exists: remove.
            SkinModel model = _modelsManager.getModel(modelName);
            if (model != null)
            {
                if (model.isModifiable())
                {
                    PathUtils.deleteDirectory(model.getPath());
                }
                else
                {
                    throw new IllegalStateException("The skin model '" + modelName + "' already exists and is not modifiable and thus cannot be replaced.");       
                }
            }
            
            // Move to models
            Path rootLocation = _modelsManager.getLocalModelsLocation();
            PathUtils.moveDirectory(tmpDir, rootLocation.resolve(modelName));
            
            _i18nUtils.reloadCatalogues();
            I18nTransformer.needsReload();
        }
        
        return modelName;
    }
    
    /**
     * Generate a new skin from a model
     * @param skinId The new skin id
     * @param modelId The model
     * @return An error message, or null on success
     * @throws IOException if an error occurs when manipulating files
     * @throws ProcessingException if an exception occurred during the generation processs 
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public String generateSkin(String skinId, String modelId) throws IOException, ProcessingException
    {
        // Check if exists
        if (_skinsManager.getSkins().contains(skinId))
        {
            return "already-exists";
        }
        
        Path modelDir = _modelsManager.getModel(modelId).getPath();
        Path skinDir = _skinsManager.getLocalSkinsLocation().resolve(skinId);
        
        // Copy the model
        PathUtils.copyDirectory(modelDir, skinDir, FileFilter.getModelFilter(modelDir), false);
        
        try
        {
            Skin skin = _skinsManager.getSkin(skinId);

            // Create model.xml file
            _modelsManager.generateModelFile(skinDir, modelId);
            
            SkinModel model = _modelsManager.getModel(modelId);
            String defaultColorTheme = model.getDefaultColorTheme();
            if (defaultColorTheme != null)
            {
                _skinFactoryManager.saveColorTheme(skin.getRawPath(), defaultColorTheme);
            }

            // Apply all parameters
            _skinFactoryManager.applyModelParameters(modelId, skin.getRawPath());
            
            I18nTransformer.needsReload();
            _i18nUtils.reloadCatalogues();
        }
        catch (Exception e)
        {
            // Delete skin directory if the generation failed
            FileUtils.deleteDirectory(skinDir.toFile());
            
            throw new ProcessingException("The generation of skin failed", e);
            
        }
        
        return null;
    }
    
    /**
     * Apply the model to all its skins
     * @param modelId The model id
     * @return The set of modified skins id
     * @throws IOException if an error occurs when manipulating files 
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public Map<String, Object> applyModelToAll(String modelId) throws IOException
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("modifiedSkins", new ArrayList<>());
        result.put("unmodifiedSkins", new ArrayList<>());
        result.put("unmodifiableSkins", new ArrayList<>());
        
        Set<String> skins = _skinsManager.getSkins();
        
        for (String skinId : skins)
        {
            Skin skin = _skinsManager.getSkin(skinId);
            if (modelId.equals(_modelsManager.getModelOfSkin(skin)))
            {
                if (!skin.isModifiable())
                {
                    @SuppressWarnings("unchecked")
                    List<Map<String, Object>> unmodifiableSkins = (List<Map<String, Object>>) result.get("unmodifiableSkins");
                    unmodifiableSkins.add(_getSkinProperty(skin));
                }
                else if (applyModel(skin, modelId))
                {
                    @SuppressWarnings("unchecked")
                    List<Map<String, Object>> modifiedSkins = (List<Map<String, Object>>) result.get("modifiedSkins");
                    modifiedSkins.add(_getSkinProperty(skin));
                }
                else
                {
                    @SuppressWarnings("unchecked")
                    List<Map<String, Object>> unmodifiedSkins = (List<Map<String, Object>>) result.get("unmodifiedSkins");
                    unmodifiedSkins.add(_getSkinProperty(skin));
                }
            }
        }
        
        return result;
    }
    
    private Map<String, Object> _getSkinProperty(Skin skin)
    {
        Map<String, Object> info = new HashMap<>();
        info.put("name", skin.getId());
        info.put("label", skin.getLabel());
        return info;
    }
    
    /**
     * Apply model to the skin
     * @param skinId The skin id
     * @param modelId The id of model
     * @return true if the model was applyed successfully
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public boolean applyModel(String skinId, String modelId) throws IOException
    {
        Skin skin = _skinsManager.getSkin(skinId);
        
        return applyModel(skin, modelId);
    }
     
    /**
     * Apply model to the skin
     * @param skin The skin
     * @param modelId The id of model
     * @return true if the model was applyed successfully
     * @throws IOException if an error occurs when manipulating files
     */
    protected boolean applyModel(Skin skin, String modelId) throws IOException
    {
        if (!skin.isModifiable())
        {
            throw new IllegalStateException("The skin '" + skin.getId() + "' is not modifiable and thus the model can not be applied.");
        }
        
        Path skinDir = skin.getRawPath();
        
        // Prepare skin in temporary file
        Path tmpDir = skinDir.getParent().resolve(skin.getId() + "." + _DATE_FORMAT.format(new Date()));
        
        // Copy the model
        Path modelDir = _modelsManager.getModel(modelId).getPath();
        PathUtils.copyDirectory(modelDir, tmpDir, FileFilter.getModelFilter(modelDir), false);
        
        // Copy upload images if exists
        Path uploadDir = skinDir.resolve("model/_uploads");
        if (Files.exists(uploadDir))
        {
            Path tmpUploadDir = tmpDir.resolve("model/_uploads");
            Files.createDirectories(tmpUploadDir);
            FileUtils.copyDirectory(uploadDir.toFile(), tmpUploadDir.toFile());
        }
        
        // Copy model.xml file
        Path xmlFile = skinDir.resolve("model.xml");
        FileUtils.copyFileToDirectory(xmlFile.toFile(), tmpDir.toFile());
        Path tmpXmlFile = tmpDir.resolve("model.xml");
        
        // Apply parameters
        _skinFactoryManager.applyModelParameters(modelId, tmpDir);
        _skinFactoryManager.updateHash(tmpXmlFile, _modelsManager.getModelHash(modelId));
        
        if (!_skinHelper.deleteQuicklyDirectory(skinDir))
        {
            getLogger().error("Cannot delete skin directory {}", skinDir.toAbsolutePath().toString()); 
            return false;
        }
        
        FileUtils.moveDirectory(tmpDir.toFile(), skinDir.toFile());
        return true;
    }
    
    /**
     * Delete a model
     * @param modelId The model id
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public void delete(String modelId) throws IOException
    {
        SkinModel model = _modelsManager.getModel(modelId);
        
        if (!model.isModifiable())
        {
            throw new IllegalStateException("The skin model '" + modelId + "' is not modified and thus cannot be removed.");       
        }
        
        // Unlink skins
        Set<String> skins = _skinsManager.getSkins();
        for (String skinId : skins)
        {
            Skin skin = _skinsManager.getSkin(skinId);
            if (skin.isModifiable() && modelId.equals(_modelsManager.getModelOfSkin(skin)))
            {
                _unlinkModel(skin);
            }
        }
        
        Path file = model.getPath();
        if (Files.exists(file))
        {
            PathUtils.deleteDirectory(file);
        }
    }

    /**
     * Unlink the skin from its model
     * @param skinId The id of the skin
     * @param modelId The id of the model
     * @return An error code, or null on success
     * @throws IOException If an error occurred while removing the link
     */
    @Callable (rights = "Web_Rights_Admin_Skins", context = "/admin")
    public String unlinkModel(String skinId, String modelId) throws IOException
    {
        Skin skin = _skinsManager.getSkin(skinId);
        
        if (!modelId.equals(_modelsManager.getModelOfSkin(skin)))
        {
            return "incorrect-model";
        }
        
        _unlinkModel(skin);
        
        return null;
    }

    private void _unlinkModel(Skin skin) throws IOException
    {
        if (!skin.isModifiable())
        {
            throw new IllegalStateException("The skin '" + skin.getId() + "' is not modifiable and thus it can not be unlink to its model.");
        }
        
        Path skinDir = skin.getRawPath();
        
        Path modelFile = skinDir.resolve("model.xml");
        Path bakFile = skinDir.resolve("model.xml.bak");
        
        // Delete old bak file if exists
        Files.deleteIfExists(bakFile);
        
        if (Files.exists(modelFile))
        {
            Files.move(modelFile, bakFile);
        }
    }
}
