/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.parameters;

import java.nio.file.Path;
import java.util.regex.Pattern;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.skinfactory.SkinFactoryComponent;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModel.Theme;

/**
 * Implementation of {@link AbstractSkinParameter} for a CSS property of type color (color, background-color, border-color)
 */
public class CSSColorParameter extends CSSParameter
{
    private static final Pattern __HEX_HASH_LESS_PATTERN = Pattern.compile("^([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$");
    private static final Pattern __HEX_PATTERN = Pattern.compile("^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$");
    
    private SkinModel _model;
    private SkinFactoryComponent _skinFactory;
    
    /**
     * Constructor
     * @param id the unique id
     * @param label the label
     * @param description the description
     * @param cssFile the css file
     * @param cssProperty the css property
     * @param defaultValue the default value
     * @param model the model
     * @param skinFactory the skin factory manager
     */
    public CSSColorParameter(String id, I18nizableText label, I18nizableText description, Path cssFile, String cssProperty, String defaultValue, SkinModel model, SkinFactoryComponent skinFactory)
    {
        super(id, label, description, cssFile, cssProperty, defaultValue);
        _model = model;
        _skinFactory = skinFactory;
    }
    
    @Override
    public void apply(Path tempDir, Path modelDir, Object value, String lang) throws SkinParameterException
    {
        String cValue = (String) value;
        
        if (cValue.startsWith("color-theme-"))
        {
            String themeId = _skinFactory.getColorTheme(tempDir);
            Theme theme = _model.getTheme(themeId);
            
            if (theme != null)
            {
                int colorIndex = Integer.parseInt(cValue.substring("color-theme-".length()));
                
                String color = theme.getColor(colorIndex);
                if (__HEX_HASH_LESS_PATTERN.matcher(color).matches())
                {
                    cValue = "#" + color.toUpperCase();
                }
                else if (__HEX_PATTERN.matcher(color).matches())
                {
                    cValue = color.toUpperCase();
                }
                else
                {
                    cValue = color; // rgb(a)
                }
            }
        }
        
        super.apply(tempDir, modelDir, cValue.equals("transparent") ? cValue : cValue, lang);
    }
}
