/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.parameters;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.io.IOUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.skin.SkinModel;

/**
 * Implementation of {@link AbstractSkinParameter} for an image
 */
public class ImageParameter extends AbstractSkinParameter
{
    private String _iconGlyph;
    private String _iconSmall;
    private String _iconLarge;
    private boolean _localUploadEnabled;
    private String _imagePath;
    
    /**
     * Constructor
     * @param relPath the relative path of the target image
     * @param label the label
     * @param description the description
     */
    public ImageParameter(String relPath, I18nizableText label, I18nizableText description)
    {
        super(relPath.replaceAll("\\\\", "/"), label, description);
        _imagePath = relPath.replaceAll("\\\\", "/");
        _localUploadEnabled = false;
    }
    
    /**
     * Constructor
     * @param relPath the relative path of the target image
     * @param label the label
     * @param description the description
     * @param iconGlyph The CSS classe for icon, to use instead of small and large icon
     * @param iconSmall The small icon
     * @param iconLarge The large icon
     */
    public ImageParameter(String relPath, I18nizableText label, I18nizableText description, String iconGlyph, String iconSmall, String iconLarge)
    {
        super(relPath.replaceAll("\\\\", "/"), label, description);
        _imagePath = relPath.replaceAll("\\\\", "/");
        _localUploadEnabled = false;
        _iconGlyph = iconGlyph;
        _iconLarge = iconLarge;
        _iconSmall = iconSmall;
    }
    
    @Override
    public SkinParameterType getType()
    {
        return SkinParameterType.IMAGE;
    }
    
    /**
     * Determines if local upload is enabled
     * @return true if local upload is enabled
     */
    public boolean isLocalUploadEnabled ()
    {
        return _localUploadEnabled;
    }
    
    /**
     * Get relative path of images
     * @return the relative path of images
     */
    public String getLibraryPath ()
    {
        return this._imagePath;
    }
    
    /**
     * Set the CSS icon
     * @param iconGlyph the CSS icon
     */
    public void setIconGlyph(String iconGlyph)
    {
        _iconGlyph = iconGlyph;
    }
    
    /**
     * Get the CSS icon
     * @return The CSS icon
     */
    public String getIconGlyph ()
    {
        return _iconGlyph;
    }
    
    /**
     * Set the small icon relative path
     * @param iconSmall the relative path of the small icon
     */
    public void setIconSmall(String iconSmall)
    {
        _iconSmall = iconSmall;
    }
    
    /**
     * Get the small icon
     * @return The small icon
     */
    public String getIconSmall ()
    {
        return _iconSmall;
    }
    
    /**
     * Set the large icon relative path
     * @param iconLarge the relative path of the large icon
     */
    public void setIconLarge(String iconLarge)
    {
        _iconLarge = iconLarge;
    }
    
    /**
     * Get the large icon
     * @return The large icon
     */
    public String getIconLarge ()
    {
        return _iconLarge;
    }
    
    @Override
    public void apply(Path tempDir, Path modelDir, Object value, String lang)
    {
        Path targetFile = tempDir.resolve("resources/img/" + this._imagePath);
        
        boolean uploaded = value instanceof FileValue ? ((FileValue) value).isUploaded() : false;
        Path libraryFile = _getLibraryFile(tempDir, modelDir, uploaded);
        
        String filePath = value instanceof FileValue ? ((FileValue) value).getPath() : (String) value;
        Path srcFile = libraryFile.resolve(filePath);
        
        _copyFile(srcFile, targetFile);
        
        Path resourcesFile = tempDir.resolve("resources");
        try
        {
            // Update css file last modified date to avoid cache issue in case of background image
            for (Path file : _listCSSFiles(resourcesFile))
            {
                Files.setLastModifiedTime(file, FileTime.fromMillis(System.currentTimeMillis()));
            }
        }
        catch (IOException e)
        {
            throw new RuntimeException("An error occurred while retrieving CSS files in " + resourcesFile.toString());
        }
    }
    
    private List<Path> _listCSSFiles(Path file) throws IOException
    {
        return Files.walk(file)
                    .filter(Files::isRegularFile)
                    .filter(f -> f.getFileName().toString().endsWith(".css"))
                    .collect(Collectors.toList());
    }
    private Path _getLibraryFile(Path tempDir, Path modelDir, boolean uploaded)
    {
        if (uploaded)
        {
            return tempDir.resolve("model/_uploads/" + this._imagePath);
        }
        else
        {
            return modelDir.resolve("model/images/" + this._imagePath);
        }
    }
    
    private void _copyFile(Path srcFile, Path targetFile)
    {
        if (!Files.exists(srcFile))
        {
            return;
        }
        
        try
        {
            Files.createDirectories(targetFile.getParent());
         
            try (InputStream is = Files.newInputStream(srcFile); OutputStream os = Files.newOutputStream(targetFile))
            {
                IOUtils.copy(is, os);
                
                Files.setLastModifiedTime(targetFile, FileTime.fromMillis(System.currentTimeMillis()));
            }
        }
        catch (IOException e)
        {
            throw new SkinParameterException ("Unable to apply image parameter '" + getId() + "'", e);
        }
    }
    
    @Override
    public void toSAX(ContentHandler contentHandler, String modelName) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", getId());
        attrs.addCDATAAttribute("type", SkinParameterType.IMAGE.name().toLowerCase());
        
        XMLUtils.startElement(contentHandler, "parameter", attrs);
        
        getLabel().toSAX(contentHandler, "label");
        getDescription().toSAX(contentHandler, "description");
        XMLUtils.createElement(contentHandler, "path", this._imagePath);
        
        if (getIconGlyph() != null)
        {
            XMLUtils.createElement(contentHandler, "iconGlyph", getIconGlyph());
        }
        
        if (getIconSmall() != null)
        {
            XMLUtils.createElement(contentHandler, "iconSmall", "/plugins/skinfactory/" + modelName + "/_thumbnail/16/16/model/images/" + this._imagePath + "/" + getIconSmall());
        }
        else
        {
            XMLUtils.createElement(contentHandler, "iconSmall", "/plugins/skinfactory/resources/img/button/image_16.png");
        }
        
        if (getIconLarge() != null)
        {
            XMLUtils.createElement(contentHandler, "iconLarge", "/plugins/skinfactory/" + modelName + "/_thumbnail/32/32/model/images/" + this._imagePath + "/" + getIconLarge());
        }
        else
        {
            XMLUtils.createElement(contentHandler, "iconLarge", "/plugins/skinfactory/resources/img/button/image_32.png");
        }
        
        XMLUtils.endElement(contentHandler, "parameter");
    }
    
    @Override
    public Map<String, Object> toJson(String modelName)
    {
        Map<String, Object> jsonObject = new HashMap<>();
        
        jsonObject.put("id", getId());
        jsonObject.put("type", SkinParameterType.IMAGE.name().toLowerCase());
        
        jsonObject.put("label", getLabel());
        jsonObject.put("description", getDescription());
        jsonObject.put("path", this._imagePath);
        
        if (getIconGlyph() != null)
        {
            jsonObject.put("iconGlyph", getIconGlyph());
        }
        
        if (getIconSmall() != null)
        {
            jsonObject.put("iconSmall", "/plugins/skinfactory/" + modelName + "/_thumbnail/16/16/model/images/" + this._imagePath + "/" + getIconSmall());
        }
        else
        {
            jsonObject.put("iconSmall", "/plugins/skinfactory/resources/img/button/image_16.png");
        }
        
        if (getIconLarge() != null)
        {
            jsonObject.put("iconLarge", "/plugins/skinfactory/" + modelName + "/_thumbnail/32/32/model/images/" + this._imagePath + "/" + getIconLarge());
        }
        else
        {
            jsonObject.put("iconLarge", "/plugins/skinfactory/resources/img/button/image_32.png");
        }
        
        return jsonObject;
    }
    
    
    @Override
    public FileValue getDefaultValue(SkinModel model)
    {
        Path libraryFile = model.getPath().resolve("model/images/" + this._imagePath);
        
        try (Stream<Path> s = Files.list(libraryFile))
        {
            List<Path> files = s.collect(Collectors.toList());
            for (Path file : files)
            {
                if (Files.isDirectory(file))
                {
                    try (Stream<Path> s2 = Files.list(file))
                    {
                        List<Path> subfiles = s2.collect(Collectors.toList());
                        for (Path child : subfiles)
                        {
                            if (_isImage(child))
                            {
                                String defaultPath = libraryFile.relativize(child).toString();
                                defaultPath.replaceAll(File.separator.equals("\\") ? File.separator + File.separator : File.separator, ".");
                                
                                return new FileValue(defaultPath, false);
                            }
                        }
                    }
                    
                }
                else if (_isImage(file))
                {
                    String defaultPath = libraryFile.relativize(file).toString();
                    defaultPath.replaceAll(File.separator.equals("\\") ? File.separator + File.separator : File.separator, ".");
                    
                    return new FileValue(defaultPath, false);
                }
            }
        }
        catch (IOException e)
        {
            throw new RuntimeException("Cannot get default value for model " + model.getId(), e);
        }
        
        return null;
    }
    
    private boolean _isImage(Path file)
    {
        if (Files.isDirectory(file))
        {
            return false;
        }
        
        String name = file.getFileName().toString().toLowerCase();
        int index = name.lastIndexOf(".");
        String ext = name.substring(index + 1);
        
        if (name.equals("thumbnail_16.png") || name.equals("thumbnail_32.png") || name.equals("thumbnail_48.png"))
        {
            return false;
        }

        return "png".equals(ext) || "gif".equals(ext) || "jpg".equals(ext) || "jpeg".equals(ext);
    }
    
    @Override
    public FileValue getDefaultValue(SkinModel model, String lang)
    {
        return getDefaultValue(model);
    }
    
    /**
     * Class representing a file value
     *
     */
    public static class FileValue 
    {
        private String _path;
        private boolean _uploaded;
        
        /**
         * Constructor
         * @param path The relative file path
         * @param uploaded <code>true</code> if the file was uploaded
         */
        public FileValue(String path, boolean uploaded)
        {
            _uploaded = uploaded;
            _path = path;
        }
        
        /**
         * Determines if the file was uploaded
         * @return <code>true</code> if the file was uploaded
         */
        public boolean isUploaded ()
        {
            return _uploaded;
        }
        
        /**
         * Get the relative file path
         * @return the relative file path
         */
        public String getPath ()
        {
            return _path;
        }
    }

}
