/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.survey.dao.SurveyDAO;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.web.repository.site.SiteManager;

/**
 * SAX events for surveys, survey pages and survey questions
 *
 */
public class GetSurveysAction extends ServiceableAction
{
    /** The String representing the type of a survey node */
    protected static final String SURVEY_TYPE = "survey";
    /** The String representing the type of a survey page node */
    protected static final String SURVEY_PAGE_TYPE = "page";
    /** The String representing the type of a survey question node */
    protected static final String SURVEY_QUESTION_TYPE = "question";
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The site manager */
    protected SiteManager _siteManager;
    /** The survey DAO */
    protected SurveyDAO _surveyDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _surveyDAO = (SurveyDAO) serviceManager.lookup(SurveyDAO.ROLE);
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String nodeId = (String) jsParameters.get("node");
        //String nodeType = (String) jsParameters.get("nodeType");
        String lang = (String) jsParameters.get("lang");
        String siteName = (String) jsParameters.get("siteName");
        
        boolean surveyOnly = (Boolean) jsParameters.get("surveyOnly");
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        
        try
        {
            if (StringUtils.isEmpty(nodeId) || "survey-root-node".equals(nodeId))
            {
                ModifiableTraversableAmetysObject rootNode = getSurveyRootNode(siteName, lang);
                
                AmetysObjectIterable<Survey> surveys = rootNode.getChildren();
                for (Survey survey : surveys)
                {
                    nodes.add(surveyToJSON(survey, surveyOnly));
                }
            }
            else if (!surveyOnly)
            {
                AmetysObject object = _resolver.resolveById(nodeId);
                if (object instanceof Survey)
                {
                    for (AmetysObject page : ((Survey) object).getPages())
                    {
                        nodes.add(surveyPageToJSON((SurveyPage) page));
                    }
                }
                else if (object instanceof SurveyPage)
                {
                    for (AmetysObject question : ((SurveyPage) object).getChildren())
                    {
                        nodes.add(surveyQuestionToJSON((SurveyQuestion) question));
                    }
                }
            }
        }
        catch (RepositoryException e)
        {
            throw new ProcessingException("Unable to get surveys", e);
        }
        
        Map<String, Object> result = new HashMap<>();
        result.put("children", nodes);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
    
    /**
     * Gets survey's properties to JSON format.
     * @param survey The survey.
     * @param surveyOnly true to display survey only
     * @return The survey's properties
     */
    protected Map<String, Object> surveyToJSON(Survey survey, boolean surveyOnly)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", survey.getId());
        infos.put("type", SURVEY_TYPE);
        infos.put("name", survey.getName());
        infos.put("private", _surveyDAO.isPrivate(survey));
        infos.put("validated", survey.isValidated());
        infos.put("title", survey.getTitle());
        infos.put("label", survey.getLabel());
        infos.put("description", survey.getDescription());
        infos.put("endingMessage", survey.getEndingMessage());
        infos.put("hasChildren", !surveyOnly && survey.getPages().size() > 0);
        
        return infos;
    }
    
    /**
     * Gets survey page's properties to JSON format.
     * @param page The survey page
     * @return The page's properties
     */
    protected Map<String, Object> surveyPageToJSON(SurveyPage page)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", page.getId());
        infos.put("type", SURVEY_PAGE_TYPE);
        infos.put("label", page.getLabel());
        infos.put("title", page.getTitle());
        infos.put("description", page.getDescription());
        infos.put("hasChildren", page.getChildren().getSize() > 0);
        
        return infos;
    }
    
    /**
     * Gets survey question's properties to JSON format.
     * @param question The survey question
     * @return The question's properties
     */
    protected Map<String, Object> surveyQuestionToJSON(SurveyQuestion question)
    {
        Map<String, Object> infos = new HashMap<>();
        
        infos.put("id", question.getId());
        infos.put("type", SURVEY_QUESTION_TYPE);
        infos.put("name", question.getName());
        infos.put("questionType", question.getType().name());
        infos.put("mandatory", question.isMandatory());
        infos.put("validated", question.getSurvey().isValidated());
        infos.put("label", question.getLabel());
        infos.put("title", question.getTitle());
        infos.put("hasChildren", false);
        
        return infos;
    }
    
    /**
     * Get the root node for surveys
     * @param siteName the site name
     * @param lang the language
     * @return the root node
     * @throws RepositoryException if an error occurs when manipulating the repository
     */
    protected ModifiableTraversableAmetysObject getSurveyRootNode (String siteName, String lang) throws RepositoryException
    {
        ModifiableTraversableAmetysObject pluginsNode = _siteManager.getSite(siteName).getRootPlugins();
        
        ModifiableTraversableAmetysObject surveyNode = null;
        if (!pluginsNode.hasChild("survey"))
        {
            surveyNode = ((ModifiableTraversableAmetysObject) pluginsNode.createChild("survey", "ametys:unstructured")).createChild("ametys:surveys", "ametys:unstructured");
        }
        else
        {
            surveyNode = pluginsNode.getChild("survey/ametys:surveys");
        }
        
        if (!surveyNode.hasChild(lang))
        {
            surveyNode.createChild(lang, "ametys:unstructured");
            ((JCRAmetysObject) pluginsNode).getNode().getSession().save();
        }
        
        return pluginsNode.getChild("survey/ametys:surveys/" + lang);
    }
}
