/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.clientsideelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.web.repository.page.Page;

/**
 * This element creates a button for page redirection of a survey
 */
public class RedirectPageClientSideElement extends StaticClientSideElement
{
    /** Repository content */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get the Redirect page status for a given survey
     * @param surveyId The survey id
     * @return The redirect page status
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getStatus(String surveyId)
    {
        Map<String, Object> results = new HashMap<>();
        
        Survey survey = _resolver.resolveById(surveyId);
        
        String redirectPageId = survey.getRedirection();
        if (StringUtils.isNotEmpty(redirectPageId))
        {
            try
            {
                Page page = _resolver.resolveById(redirectPageId);
                
                List<String> i18nParameters = new ArrayList<>();
                i18nParameters.add(page.getTitle());
                i18nParameters.add(page.getSitemapName() + "/" + page.getPathInSitemap() + ".html");
                
                I18nizableText ed = (I18nizableText) this._script.getParameters().get("redirect-page-description");
                I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                
                results.put("page-redirection", redirectPageId);
                results.put("page-redirection-description", msg);
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("The redirect page of id '" + redirectPageId + "' does not exist anymore", e);
            }
        }
        
        return results;
    }

}
