/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.survey.SurveyEvents;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.plugins.survey.repository.SurveyQuestion.QuestionType;
import org.ametys.plugins.survey.repository.SurveyRule;
import org.ametys.plugins.survey.repository.SurveyRule.RuleType;

/**
 * DAO for manipulating survey questions.
 *
 */
public class QuestionDAO extends AbstractDAO
{
    /** The Avalon role */
    public static final String ROLE = QuestionDAO.class.getName();
    
    /**
     * Gets properties of a survey question
     * @param id The id of the survey question
     * @return The properties
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getQuestion (String id)
    {
        SurveyQuestion question = _resolver.resolveById(id);
        
        return getQuestion(question);
    }
    
    /**
     * Gets properties of a survey question
     * @param question The survey question
     * @return The properties
     */
    public Map<String, Object> getQuestion (SurveyQuestion question)
    {
        Map<String, Object> properties = new HashMap<>();
        
        properties.put("id", question.getId());
        properties.put("type", question.getType());
        properties.put("label", question.getLabel());
        properties.put("title", question.getTitle());
        properties.put("regexp", question.getRegExpType());
        properties.put("mandatory", String.valueOf(question.isMandatory()));
        properties.put("validated", String.valueOf(question.getSurvey().isValidated()));
        
        Map<String, String> columns = question.getColumns();
        if (columns.size() > 0)
        {
            properties.put("columns", columns);
        }
        
        Map<String, String> options = question.getOptions();
        if (options.size() > 0)
        {
            properties.put("otherOption", String.valueOf(question.hasOtherOption()));
            properties.put("options", options);
        }
        
        properties.putAll(getPictureInfo(question));
        
        return properties;
    }
    
    /**
     * Creates a survey question.
     * @param values The question's values
     * @return The id of the created survey question, the id of the page and the id of the survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> createQuestion (Map<String, Object> values)
    {
        Map<String, String> result = new HashMap<>();
        
        String pageId = StringUtils.defaultString((String) values.get("pageId"));
        SurveyPage page = _resolver.resolveById(pageId);
        
        String label = StringUtils.defaultString((String) values.get("label"));
        String originalName = NameHelper.filterName(label);
        
        // Find unique name
        String uniqueName = page.getSurvey().findUniqueQuestionName(originalName);
        
        SurveyQuestion question = page.createChild(uniqueName, "ametys:survey-question");
        
        _setValues(question, values);
        
        page.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", page.getSurvey());
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", question.getId());
        result.put("pageId", page.getId());
        result.put("surveyId", question.getSurvey().getId());
        result.put("type", question.getType().toString());
        
        return result;
    }
    
    /**
     * Edits a survey question.
     * @param values The question's values
     * @return The id of the edited survey question, the id of the page and the id of the survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> editQuestion (Map<String, Object> values)
    {
        Map<String, String> result = new HashMap<>();
        
        String id = StringUtils.defaultString((String) values.get("id"));
        SurveyQuestion question = _resolver.resolveById(id);
        
        _setValues(question, values);
        
        question.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", question.getSurvey());
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", question.getId());
        result.put("pageId", question.getParent().getId());
        result.put("surveyId", question.getSurvey().getId());
        result.put("type", question.getType().toString());
        
        return result;
    }
    
    private void _setValues (SurveyQuestion question, Map<String, Object> values)
    {
        question.setTitle(StringUtils.defaultString((String) values.get("title")));
        question.setLabel(StringUtils.defaultString((String) values.get("label")));
        question.setMandatory("true".equals(values.get("mandatory")));
        
        String qType = StringUtils.defaultString((String) values.get("type"));
        question.setType(QuestionType.valueOf(qType));
        question.setRegExpType(StringUtils.defaultString((String) values.get("regexp")));
        
        question.setPictureAlternative(StringUtils.defaultString((String) values.get("picture-alternative")));
        setPicture(question, StringUtils.defaultString((String) values.get("picture")));
        
        boolean hasOtherOption = "true".equals(values.get("otherOption"));
        
        String options = (String) values.get("options");
        if (StringUtils.isNotEmpty(options))
        {
            Map<String, Object> rawOpts = _jsonUtils.convertJsonToMap(options);
            Map<String, String> opts = new LinkedHashMap<>();
            
            for (String optLabel : rawOpts.keySet())
            {
                String optVal = (String) rawOpts.get(optLabel);
                if (StringUtils.isEmpty(optVal))
                {
                    optVal = NameHelper.filterName(SurveyQuestion.OPTION_NAME_PREFIX + optLabel);
                }
                opts.put(optVal, optLabel);
            }
            question.setOptions(opts);
            question.setOtherOption(hasOtherOption);
        }
        
        String columns = (String) values.get("columns");
        if (StringUtils.isNotEmpty(columns))
        {
            Map<String, Object> rawCols = _jsonUtils.convertJsonToMap(columns);
            Map<String, String> cols = new LinkedHashMap<>();
            
            for (String colLabel : rawCols.keySet())
            {
                String colVal = (String) rawCols.get(colLabel);
                if (StringUtils.isEmpty(colVal))
                {
                    colVal = NameHelper.filterName(SurveyQuestion.OPTION_NAME_PREFIX + colLabel);
                }
                cols.put(colVal, colLabel);
            }
            
            question.setColumns(cols);
        }
    }
    
    /**
     * Copies and pastes a survey question.
     * @param pageId The id of the page, target of the copy
     * @param questionId The id of the question to copy
     * @return The id of the created question, the id of the page and the id of the survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> copyQuestion(String pageId, String questionId)
    {
        Map<String, String> result = new HashMap<>();
        
        SurveyQuestion originalQuestion = _resolver.resolveById(questionId);
        SurveyPage parentPage = _resolver.resolveById(pageId);
        
        Survey originalSurvey = originalQuestion.getSurvey();
        Survey parentSurvey = parentPage.getSurvey();
        
        String name = parentSurvey.findUniqueQuestionName(originalQuestion.getName());
        SurveyQuestion cQuestion = originalQuestion.copyTo(parentPage, name);
        
        if (!originalSurvey.getId().equals(parentSurvey.getId()))
        {
            // Update rules references after copy
            updateReferencesAfterCopy (originalSurvey, parentSurvey, cQuestion);
        }
        
        parentPage.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", parentSurvey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", cQuestion.getId());
        result.put("pageId", parentPage.getId());
        result.put("surveyId", parentSurvey.getId());
        result.put("type", cQuestion.getType().toString());
        
        return result;
    }
    
    /**
     * Deletes a survey question.
     * @param id The id of the survey question to delete
     * @return The id of the deleted survey question, the id of the page and the id of the survey
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> deleteQuestion (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        SurveyQuestion question = _resolver.resolveById(id);
        String type = question.getType().toString();
        
        SurveyPage page = question.getParent();
        question.remove();
        
        page.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", page.getSurvey());
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", id);
        result.put("pageId", page.getId());
        result.put("surveyId", page.getSurvey().getId());
        result.put("type", type);
        
        return result;
    }
    
    /**
     * Adds a new rule to a question.
     * @param id _resolver.resolveById(id);
     * @param option The option
     * @param rule The rule type
     * @param page The page to jump or skip
     * @return An empty map, or an error
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> addRule (String id, String option, String rule, String page)
    {
        Map<String, Object> result = new HashMap<>();
        
        SurveyQuestion question = _resolver.resolveById(id);
        // Check if exists
        if (question.hasRule(option))
        {
            result.put("error", "already-exists");
            return result;
        }
        
        question.addRules(option, RuleType.valueOf(rule), page);
        question.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", question.getSurvey());
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        return result;
    }
    
    /**
     * Deletes a rule to a question.
     * @param id _resolver.resolveById(id);
     * @param option The option to delete
     * @return An empty map
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> deleteRule (String id, String option)
    {
        SurveyQuestion question = _resolver.resolveById(id);
        
        question.deleteRule(option);
        question.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", question.getSurvey());
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        return new HashMap<>();
    }
    
    /**
     * Gets the rules for a survey question.
     * @param id The id of the survey question.
     * @param number The question number
     * @return The rules
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getRules (String id, int number)
    {
        Map<String, Object> result = new HashMap<>();
        
        SurveyQuestion question = _resolver.resolveById(id);
        Map<String, String> options = question.getOptions();
        
        result.put("id", question.getId());
        result.put("number", String.valueOf(number));
        result.put("title", question.getTitle());
        
        List<Object> rules = new ArrayList<>();
        for (SurveyRule rule : question.getRules())
        {
            Map<String, Object> resultRule = new HashMap<>();
            String option = rule.getOption();
            resultRule.put("option", option);
            resultRule.put("optionLabel", options.get(option));
            resultRule.put("type", rule.getType().name());
            String page = rule.getPage();
            if (page != null)
            {
                try
                {
                    SurveyPage pageAO = _resolver.resolveById(page);
                    resultRule.put("page", page);
                    resultRule.put("pageName", pageAO.getLabel());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Page does not exist anymore
                }
            }
                    
            rules.add(resultRule);
        }
        
        result.put("rules", rules);
        
        return result;
    }

}
