/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.generators;

import java.io.IOException;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.survey.SurveyDateUtils;
import org.ametys.plugins.survey.data.SurveyAnswerDao;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;

/**
 * SAX surveys
 *
 */
public class SurveyGenerator extends SurveysGenerator
{
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    /** The current user provider. */
    protected SurveyAnswerDao _surveyAnswerDao;
    /** The user provider. */
    protected CurrentUserProvider _userProvider;
    private RightManager _rightManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _surveyAnswerDao = (SurveyAnswerDao) serviceManager.lookup(SurveyAnswerDao.ROLE);
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String id = parameters.getParameter("id", request.getParameter("id"));
        
        contentHandler.startDocument();
        
        Survey survey = _resolver.resolveById(id);
        
        Calendar now = new GregorianCalendar();
        now.set(Calendar.HOUR_OF_DAY, 0);
        now.set(Calendar.MINUTE, 0);
        now.set(Calendar.SECOND, 0);
        now.set(Calendar.MILLISECOND, 0);
        
        Date startDate = survey.getStartDate();
        Date endDate = survey.getEndDate();

        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", survey.getId());
        attrs.addCDATAAttribute("name", survey.getName());
        attrs.addCDATAAttribute("private", String.valueOf(_surveyDAO.isPrivate(survey)));

        UserIdentity user = _userProvider.getUser();
        attrs.addCDATAAttribute("canRead", String.valueOf(_rightManager.hasReadAccess(user, survey)));
        
        attrs.addCDATAAttribute("validated", String.valueOf(survey.isValidated()));
        attrs.addCDATAAttribute("status", _getStatus(startDate, endDate));
        
        ZonedDateTime reinitDate = survey.getReinitDate();
        if (reinitDate != null)
        {
            attrs.addCDATAAttribute("reinitDate", SurveyDateUtils.zonedDateTimeToString(reinitDate));
        }
        
        XMLUtils.startElement(contentHandler, "survey", attrs);
        XMLUtils.createElement(contentHandler, "title", survey.getTitle());
        XMLUtils.createElement(contentHandler, "label", survey.getLabel());

        if (startDate != null)
        {
            XMLUtils.createElement(contentHandler, "startDate", DateUtils.dateToString(startDate));
        }
        if (endDate != null)
        {
            XMLUtils.createElement(contentHandler, "endDate", DateUtils.dateToString(endDate));
        }
        
        String description = survey.getDescription();
        if (description != null)
        {
            XMLUtils.createElement(contentHandler, "description", description);
        }
        
        String endingMessage = survey.getEndingMessage();
        if (endingMessage != null)
        {
            XMLUtils.createElement(contentHandler, "endingMessage", endingMessage);
        }
        
        for (SurveyPage page : survey.getPages())
        {
            toSAX(page, true, true);
        }
        
        toSAXPicture(survey);
        
        XMLUtils.endElement(contentHandler, "survey");
        
        contentHandler.endDocument();
    }
    
    private String _getStatus (Date startDate, Date endDate)
    {
        Calendar today = new GregorianCalendar();
        today.set(Calendar.HOUR_OF_DAY, 0);
        today.set(Calendar.MINUTE, 0);
        today.set(Calendar.SECOND, 0);
        today.set(Calendar.MILLISECOND, 0);
        
        if (startDate != null && startDate.after(today.getTime()))
        {
            return "coming";
        }
        
        if (endDate != null && endDate.before(today.getTime()))
        {
            return "over";
        }
        
        return "open";
    }
}
