/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.generators;

import java.io.IOException;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.ParameterException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.survey.data.SurveyAnswer;
import org.ametys.plugins.survey.data.SurveyAnswerDao;
import org.ametys.plugins.survey.data.SurveySession;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyQuestion;

/**
 * Generates a specific survey user session from its ID.
 */
public class SurveySessionGenerator extends ServiceableGenerator
{

    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The survey data DAO. */
    protected SurveyAnswerDao _surveyDao;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _surveyDao = (SurveyAnswerDao) serviceManager.lookup(SurveyAnswerDao.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
//        Request request = ObjectModelHelper.getRequest(objectModel);
        
        // Get the session ID.
        int sessionId;
        try
        {
            sessionId = parameters.getParameterAsInteger("id");
        }
        catch (ParameterException e)
        {
            throw new ProcessingException("Session ID is mandatory.", e);
        }
        
        // Retrieve the corresponding session from the database.
        SurveySession surveySession = _surveyDao.getSessionWithAnswers(sessionId);
        
        // Resolve the corresponding survey.
        Survey survey = _resolver.resolveById(surveySession.getSurveyId());
        
        contentHandler.startDocument();
        
        // Generate the session and its answers.
        saxSession(surveySession, survey, true);
        
        contentHandler.endDocument();
    }
    
    /**
     * Generate the data of a survey user session.
     * @param surveySession the survey session.
     * @param survey the survey.
     * @param withAnswers true to generate answers along, false otherwise.
     * @throws SAXException if an error occurs while saxing the session
     */
    protected void saxSession(SurveySession surveySession, Survey survey, boolean withAnswers) throws SAXException
    {
        ZonedDateTime submittedAt = surveySession.getSubmittedAt().toInstant().atZone(ZoneId.systemDefault());
        
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", Integer.toString(surveySession.getId()));
        attrs.addCDATAAttribute("submittedAt", DateUtils.getISODateTimeFormatter().format(submittedAt));
        UserIdentity user = surveySession.getUser();
        if (user != null)
        {
            attrs.addCDATAAttribute("user", UserIdentity.userIdentityToString(user));
        }
        
        XMLUtils.startElement(contentHandler, "session", attrs);
        
        if (withAnswers)
        {
            saxAnswers(surveySession, survey);
        }
        
        XMLUtils.endElement(contentHandler, "session");
    }

    /**
     * Generate the answers of a given session.
     * @param surveySession the survey session.
     * @param survey the survey.
     * @throws SAXException if an error occurs while saxing the answers
     */
    protected void saxAnswers(SurveySession surveySession, Survey survey) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        
        Map<String, SurveyAnswer> answerMap = getAnswerMap(surveySession);
        
        for (SurveyQuestion question : survey.getQuestions())
        {
            SurveyAnswer answer = answerMap.get(question.getName());
            
            attrs.clear();
            attrs.addCDATAAttribute("name", question.getName());
            attrs.addCDATAAttribute("title", question.getTitle());
            attrs.addCDATAAttribute("type", question.getType().toString());
            attrs.addCDATAAttribute("mandatory", Boolean.toString(question.isMandatory()));
            XMLUtils.startElement(contentHandler, "question", attrs);
            
            if (answer != null)
            {
                Map<String, Set<String>> values = getValueMap(question, answer.getValue());
                
                for (String option : values.keySet())
                {
                    attrs.clear();
                    attrs.addCDATAAttribute("name", option);
                    XMLUtils.startElement(contentHandler, "option", attrs);
                    
                    for (String value : values.get(option))
                    {
                        attrs.clear();
                        attrs.addCDATAAttribute("value", value);
                        XMLUtils.createElement(contentHandler, "answer", attrs);
                    }
                    
                    XMLUtils.endElement(contentHandler, "option");
                }
            }
            
            XMLUtils.endElement(contentHandler, "question");
        }
    }
    
    /**
     * Get the answers of a survey session as a Map, indexed by question ID.
     * @param surveySession the survey session.
     * @return the answers as a Map, indexed by question ID. 
     */
    protected Map<String, SurveyAnswer> getAnswerMap(SurveySession surveySession)
    {
        Map<String, SurveyAnswer> answerMap = new HashMap<>();
        
        for (SurveyAnswer answer : surveySession.getAnswers())
        {
            answerMap.put(answer.getQuestionId(), answer);
        }
        
        return answerMap;
    }
    
    /**
     * Get the user-input value as a Map from the database value, which is a single serialized string.
     * @param question the question.
     * @param value the value from the database.
     * @return the value as a Map.
     */
    protected Map<String, Set<String>> getValueMap(SurveyQuestion question, String value)
    {
        Map<String, Set<String>> values = new HashMap<>();
        
        if (value != null)
        {
            switch (question.getType())
            {
                case SINGLE_MATRIX:
                case MULTIPLE_MATRIX:
                    String[] entries = StringUtils.split(value, ';');
                    for (String entry : entries)
                    {
                        String[] keyValue = StringUtils.split(entry, ':');
                        if (keyValue.length == 2 && StringUtils.isNotEmpty(keyValue[0]))
                        {
                            Set<String> valueSet = new HashSet<>(Arrays.asList(StringUtils.split(keyValue[1], ',')));
                            
                            values.put(keyValue[0], valueSet);
                        }
                    }
                    break;
                case SINGLE_CHOICE:
                case MULTIPLE_CHOICE:
                    Set<String> valueSet = new HashSet<>(Arrays.asList(StringUtils.split(value, ',')));
                    values.put("values", valueSet);
                    break;
                case FREE_TEXT:
                case MULTILINE_FREE_TEXT:
                default:
                    values.put("values", Collections.singleton(value));
                    break;
            }
        }
        
        return values;
    }

}
