/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.generators;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.data.Binary;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.survey.dao.SurveyDAO;
import org.ametys.plugins.survey.repository.AbstractSurveyElement;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.plugins.survey.repository.SurveyRule;
import org.ametys.plugins.survey.repository.SurveyRule.RuleType;
import org.ametys.web.repository.site.SiteManager;

/**
 * SAX surveys
 *
 */
public class SurveysGenerator extends ServiceableGenerator
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The site manager */
    protected SiteManager _siteManager;
    /** The DAO for surveys */
    protected SurveyDAO _surveyDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _surveyDAO = (SurveyDAO) serviceManager.lookup(SurveyDAO.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String siteName = request.getParameter("siteName");
        String lang = request.getParameter("lang");
        String id = request.getParameter("id");
        boolean saxChildren = parameters.getParameterAsBoolean("saxChildren", true);
        
        contentHandler.startDocument();
        
        try
        {
            if (StringUtils.isEmpty(id) || id.equals("survey-root-node"))
            {
                XMLUtils.startElement(contentHandler, "surveys");
                
                ModifiableTraversableAmetysObject rootNode = getSurveyRootNode(siteName, lang);
                
                AmetysObjectIterable<Survey> surveys = rootNode.getChildren();
                for (Survey survey : surveys)
                {
                    toSAX(survey, false);
                }
                
                XMLUtils.endElement(contentHandler, "surveys");
            }
            else
            {
                AmetysObject ao = _resolver.resolveById(id);
                if (ao instanceof Survey)
                {
                    toSAX((Survey) ao, saxChildren);
                }
                else if (ao instanceof SurveyPage)
                {
                    toSAX((SurveyPage) ao, saxChildren, false);
                }
                else if (ao instanceof SurveyQuestion)
                {
                    toSAX((SurveyQuestion) ao);
                }
            }
        }
        catch (RepositoryException e)
        {
            throw new ProcessingException("Unable to get surveys", e);
        }
        
        contentHandler.endDocument();

    }
    
    /**
     * SAX a survey
     * @param survey the survey to SAX
     * @param saxChildren true to SAX children
     * @throws SAXException if error occurs while SAXing
     */
    protected void toSAX (Survey survey, boolean saxChildren) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", survey.getId());  
        attrs.addCDATAAttribute("name", survey.getName());
        attrs.addCDATAAttribute("private", String.valueOf(_surveyDAO.isPrivate(survey)));
        attrs.addCDATAAttribute("validated", String.valueOf(survey.isValidated()));
        
        XMLUtils.startElement(contentHandler, "survey", attrs);
        XMLUtils.createElement(contentHandler, "title", survey.getTitle());
        XMLUtils.createElement(contentHandler, "label", survey.getLabel());
        
        String description = survey.getDescription();
        if (description != null)
        {
            XMLUtils.createElement(contentHandler, "description", description);
        }
        
        String endingMessage = survey.getEndingMessage();
        if (endingMessage != null)
        {
            XMLUtils.createElement(contentHandler, "endingMessage", endingMessage);
        }
        
        toSAXPicture(survey);
        
        if (saxChildren)
        {
            AmetysObjectIterable<SurveyPage> pages = survey.getChildren();
            for (SurveyPage page : pages)
            {
                toSAX(page, false, false);
            }
        }
        
        XMLUtils.endElement(contentHandler, "survey");
    }
    
    /**
     * SAX a survey page
     * @param page the page to SAX
     * @param saxChildren true to SAX children
     * @param saxBranches true to SAX branch logic
     * @throws SAXException if error occurs while SAXing
     */
    protected void toSAX (SurveyPage page, boolean saxChildren, boolean saxBranches) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", page.getId());
        attrs.addCDATAAttribute("name", page.getName());
        
        XMLUtils.startElement(contentHandler, "page", attrs);
        XMLUtils.createElement(contentHandler, "title", page.getTitle());
        XMLUtils.createElement(contentHandler, "label", page.getLabel());
        XMLUtils.createElement(contentHandler, "description", page.getDescription());
        
        toSAXPicture(page);
        
        if (saxChildren)
        {
            AmetysObjectIterable<SurveyQuestion> questions = page.getChildren();
            for (SurveyQuestion question : questions)
            {
                toSAX(question);
            }
        }
        
        if (saxBranches)
        {
            saxBranches(page);
        }
        
        XMLUtils.endElement(contentHandler, "page");
    }
    
    /**
     * SAX branch logic
     * @param page the page
     * @throws SAXException if error occurs while SAXing
     */
    protected void saxBranches (SurveyPage page) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "branches");
        
        AmetysObjectIterable<SurveyQuestion> questions = page.getChildren();
        for (SurveyQuestion question : questions)
        {
            List<SurveyRule> rules = question.getRules();
            
            for (SurveyRule rule : rules)
            {
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("id", question.getId());
                attr.addCDATAAttribute("name", question.getName());
                attr.addCDATAAttribute("value", rule.getOption());
                RuleType type = rule.getType();
                attr.addCDATAAttribute("type", type.name());
                if (type == RuleType.JUMP)
                {
                    attr.addCDATAAttribute("page", rule.getPage());
                }
                XMLUtils.createElement(contentHandler, "rule", attr);
            }
        }
        
        SurveyRule rule = page.getRule();
        if (rule != null)
        {
            AttributesImpl attr = new AttributesImpl();
            RuleType type = rule.getType();
            attr.addCDATAAttribute("type", type.name());
            if (type == RuleType.JUMP)
            {
                attr.addCDATAAttribute("page", rule.getPage());
            }
            XMLUtils.createElement(contentHandler, "page-rule", attr);
        }
        
        XMLUtils.endElement(contentHandler, "branches");
    }
    
    /**
     * SAX a survey question
     * @param question the question to SAX
     * @throws SAXException if error occurs while SAXing
     */
    protected void toSAX (SurveyQuestion question) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", question.getId());
        attrs.addCDATAAttribute("name", question.getName());
        attrs.addCDATAAttribute("type", question.getType().name());
        attrs.addCDATAAttribute("mandatory", String.valueOf(question.isMandatory()));
        attrs.addCDATAAttribute("modifiable", String.valueOf(!question.getSurvey().isValidated()));
        
        XMLUtils.startElement(contentHandler, "question", attrs);
        XMLUtils.createElement(contentHandler, "label", question.getLabel());
        XMLUtils.createElement(contentHandler, "title", question.getTitle());
        
        toSAXPicture(question);
        
        String regExpType = question.getRegExpType();
        if (StringUtils.isNotEmpty(regExpType))
        {
            XMLUtils.createElement(contentHandler, "regexp", regExpType);
        }
        
        String pattern = question.getRegExpPattern();
        if (StringUtils.isNotEmpty(pattern))
        {
            XMLUtils.createElement(contentHandler, "pattern", pattern);
        }
        
        Map<String, String> options = question.getOptions();
        if (options.size() > 0)
        {
            AttributesImpl attr = new AttributesImpl();
            attr.addCDATAAttribute("other-option", String.valueOf(question.hasOtherOption()));
            XMLUtils.startElement(contentHandler, "options", attr);
            for (String value : options.keySet())
            {
                attr = new AttributesImpl();
                attr.addCDATAAttribute("value", value);
                XMLUtils.createElement(contentHandler, "option", attr, options.get(value));
            }
            
            XMLUtils.endElement(contentHandler, "options");
        }
        
        Map<String, String> columns = question.getColumns();
        if (columns.size() > 0)
        {
            XMLUtils.startElement(contentHandler, "columns");
            for (String value : columns.keySet())
            {
                AttributesImpl attr = new AttributesImpl();
                attr.addCDATAAttribute("value", value);
                XMLUtils.createElement(contentHandler, "column", attr, columns.get(value));
            }
            XMLUtils.endElement(contentHandler, "columns");
        }
        
        
        XMLUtils.endElement(contentHandler, "question");
    }
    
    /**
     * SAX the picture of the element of a survey
     * @param elmt The survey element
     * @throws SAXException If an error occurred
     */
    protected void toSAXPicture (AbstractSurveyElement elmt) throws SAXException
    {
        String pictureType = elmt.getPictureType();
        
        if (pictureType != null)
        {
            AttributesImpl attrsPicture = new AttributesImpl();
            attrsPicture.addCDATAAttribute("pictureType", pictureType);
            if (pictureType.equals("resource"))
            {
                String resourceId = elmt.getResourcePictureId();
                attrsPicture.addCDATAAttribute("pictureId", resourceId);
                try
                {
                    Resource resource = _resolver.resolveById(resourceId);
                    attrsPicture.addCDATAAttribute("pictureName", resource.getName());
                    attrsPicture.addCDATAAttribute("pictureSize", Long.toString(resource.getLength()));
                    attrsPicture.addCDATAAttribute("imageType", "explorer");
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().error("The resource of id'" + resourceId + "' does not exist anymore. The picture for element survey of id '" + elmt.getId() + "' will be ignored.", e);
                }
                
            }
            else if (pictureType.equals("external"))
            {
                Binary picMeta = elmt.getExternalPicture();
                attrsPicture.addCDATAAttribute("picturePath", "picture");
                attrsPicture.addCDATAAttribute("pictureName", picMeta.getFilename());
                attrsPicture.addCDATAAttribute("pictureSize", Long.toString(picMeta.getLength()));
                attrsPicture.addCDATAAttribute("imageType", "metadata");
            }
            XMLUtils.createElement(contentHandler, "picture", attrsPicture);
        }
        
        XMLUtils.createElement(contentHandler, "pictureAlternative", StringUtils.defaultString(elmt.getPictureAlternative()));
    }
    
    
    /**
     * Get the root node for surveys
     * @param siteName the site name
     * @param lang the language
     * @return the root node
     * @throws RepositoryException if an error occurs when retrieving the survey's root node
     */
    protected ModifiableTraversableAmetysObject getSurveyRootNode (String siteName, String lang) throws RepositoryException
    {
        ModifiableTraversableAmetysObject pluginsNode = _siteManager.getSite(siteName).getRootPlugins();
        
        ModifiableTraversableAmetysObject surveyNode = null;
        if (!pluginsNode.hasChild("survey"))
        {
            surveyNode = ((ModifiableTraversableAmetysObject) pluginsNode.createChild("survey", "ametys:unstructured")).createChild("ametys:surveys", "ametys:unstructured");
        }
        else
        {
            surveyNode = pluginsNode.getChild("survey/ametys:surveys");
        }
        
        if (!surveyNode.hasChild(lang))
        {
            surveyNode.createChild(lang, "ametys:unstructured");
            ((JCRAmetysObject) pluginsNode).getNode().getSession().save();
        }
        
        return pluginsNode.getChild("survey/ametys:surveys/" + lang);
    }

}
