/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.repository;

import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.Date;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.environment.Cookie;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.survey.SurveyDateUtils;
import org.ametys.plugins.survey.answer.ProcessInputAction;
import org.ametys.plugins.survey.data.SurveyAnswerDao;
import org.ametys.plugins.survey.data.SurveySession;

/**
 * helper to check survey access
 *
 */
public class SurveyAccessHelper extends AbstractLogEnabled implements Component, ThreadSafe, Serviceable
{
    /** Avalon role */
    public static final String ROLE = SurveyAccessHelper.class.getName();
    
    /** The ametys object resolver. */
    protected SurveyAnswerDao _answerDao;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _answerDao = (SurveyAnswerDao) manager.lookup(SurveyAnswerDao.ROLE);
    }
    
    /**
     * Returns the date on which the user answered to the survey or <code>null</code> if he was never answered
     * @param surveyId the survey ID.
     * @param user the user.
     * @return the date on which the survey was taken, or <code>null</code> if the user never took the survey.
     */
    public Date getSubmissionDate(String surveyId, UserIdentity user)
    {
        Date alreadyTakenOn = null;
        
        if (user != null && StringUtils.isNotBlank(user.getLogin()) && StringUtils.isNotBlank(user.getPopulationId()))
        {
            SurveySession userSession = _answerDao.getSession(surveyId, user);
            
            if (userSession != null)
            {
                alreadyTakenOn = userSession.getSubmittedAt();
            }
        }
        
        return alreadyTakenOn;
    }
    
    /**
     * Return the name of cookie if the survey was already taken or <code>null</code> otherwise.
     * @param request the request.
     * @param survey the survey.
     * @return the name of cookie if the survey was already taken, or <code>null</code> otherwise.
     */
    public String getCookieName (Request request, Survey survey)
    {
        Map<String, Cookie> cookieMap = request.getCookieMap();
        
        if (cookieMap.containsKey(ProcessInputAction.COOKIE_NAME))
        {
            Cookie cookie = cookieMap.get(ProcessInputAction.COOKIE_NAME);
            String takenSurveys = cookie.getValue();
            if (StringUtils.isNotEmpty(takenSurveys))
            {
                String surveyId = survey.getId();
                ZonedDateTime reinitDate = survey.getReinitDate();
                String[] takenSurveyCookies = StringUtils.split(takenSurveys, '|');
                // We check all cookies for the taken surveys
                for (int i = 0; i < takenSurveyCookies.length; i++)
                {
                    // A survey cookie is constructed as surveyId#creationCookieDate when the anonymous user has answered
                    String takenSurveyCookie = takenSurveyCookies[i];
                    String cookieSurveyDate = StringUtils.substringAfter(takenSurveyCookie, "#");
                    String cookieSurveyId = StringUtils.substringBefore(takenSurveyCookie, "#");
                    
                    // The survey has been reinitialized at least one time
                    if (reinitDate != null)
                    {
                        // So we compare the last re-initialization date with the date of creation of the cookie
                        try
                        {
                            ZonedDateTime cookieDate = SurveyDateUtils.parseZonedDateTime(cookieSurveyDate);
                            if (reinitDate.isBefore(cookieDate))
                            {
                                // If the re-initialization date is before the date of creation of the cookie, the cookie is valid and we check the survey id
                                if (surveyId.equals(cookieSurveyId))
                                {
                                    return ProcessInputAction.COOKIE_NAME;
                                }
                            }
                        }
                        catch (DateTimeParseException e) 
                        {
                            // The cookie is bad, don't check it
                        }
                    }
                    else if (surveyId.equals(cookieSurveyId))
                    {
                        return ProcessInputAction.COOKIE_NAME;
                    }
                    
                }
            }
        }
        
        return null;
    }
    
}
