/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.repository;

import java.util.LinkedHashMap;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Enumerator;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.SiteManager;

/**
 * Enumerates existing surveys.
 */
public class SurveyEnumerator extends AbstractLogEnabled implements Enumerator<String>, Serviceable, Contextualizable
{
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The site manager. */
    protected SiteManager _siteManager;
    
    private Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    public Map<String, I18nizableText> getEntries() throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        
        Map<String, I18nizableText> entries = new LinkedHashMap<>();
        
        String pageId = (String) request.getAttribute(WebConstants.REQUEST_ATTR_PAGE_ID);
        Page page = _resolver.resolveById(pageId);
        
        String siteName = (String) request.getAttribute("siteName");
        String language = page.getSitemapName();
        
        TraversableAmetysObject rootNode = getSurveyRootNode(siteName, language);
        
        try (AmetysObjectIterable<Survey> surveys = rootNode.getChildren())
        {
            for (Survey survey : surveys)
            {
                if (survey.isValidated())
                {
                    I18nizableText label = new I18nizableText(survey.getLabel());
                    entries.put(survey.getId(), label);
                }
            }
        }
        
        return entries;
    }
    
    public I18nizableText getEntry(String value) throws Exception
    {
        Survey survey = _resolver.resolveById(value);
        return new I18nizableText(survey.getLabel());
    }
    
    /**
     * Get the root node for surveys
     * @param siteName the site name
     * @param lang the language
     * @return the root node
     * @throws RepositoryException if an error occurs when retrieving the root node of the survey
     */
    protected ModifiableTraversableAmetysObject getSurveyRootNode(String siteName, String lang) throws RepositoryException
    {
        ModifiableTraversableAmetysObject pluginsNode = _siteManager.getSite(siteName).getRootPlugins();
        
        ModifiableTraversableAmetysObject surveyNode = null;
        if (!pluginsNode.hasChild("survey"))
        {
            surveyNode = ((ModifiableTraversableAmetysObject) pluginsNode.createChild("survey", "ametys:unstructured")).createChild("ametys:surveys", "ametys:unstructured");
            pluginsNode.saveChanges();
        }
        else
        {
            surveyNode = pluginsNode.getChild("survey/ametys:surveys");
        }
        
        if (!surveyNode.hasChild(lang))
        {
            surveyNode.createChild(lang, "ametys:unstructured");
            pluginsNode.saveChanges();
        }
        
        return pluginsNode.getChild("survey/ametys:surveys/" + lang);
    }
}
