/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.statistics;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.survey.data.SurveyAnswerDao;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugins.admin.statistics.Statistics;
import org.ametys.runtime.plugins.admin.statistics.StatisticsNode;
import org.ametys.runtime.plugins.admin.statistics.StatisticsProvider;
import org.ametys.runtime.plugins.admin.statistics.StatisticsValue;

/**
 * Send statistics about survey
 */
public class SurveyStatisticsProvider implements StatisticsProvider, Serviceable, PluginAware
{
    private String _id;
    private AmetysObjectResolver _ametysObjectResolver;
    private SurveyAnswerDao _surveyAnswerDAO;

    public void service(ServiceManager manager) throws ServiceException
    {
        _ametysObjectResolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _surveyAnswerDAO = (SurveyAnswerDao) manager.lookup(SurveyAnswerDao.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public Statistics getStatistics()
    {
        List<String> surveyIds = _getSurveyIds();
        List<Map<String, Object>> allSessionCount = _surveyAnswerDAO.getAllSessionCount();
        
        List<Long> counts = allSessionCount.stream()
                                            .filter(m -> surveyIds.contains(m.get("surveyId")))
                                            .map(m -> m.get("count(*)"))
                                            .map(Long.class::cast)
                                            .sorted()
                                            .toList();
        
        long count = counts.stream()
                            .mapToLong(l -> l)
                            .sum();

        
        return new StatisticsNode(
            _id,
            new I18nizableText("plugin.survey", "PLUGINS_SURVEY_USAGESTATISTICS_LABEL"),
            "ametysicon-desktop-clipboard-list",
            null,
            List.of(
                new StatisticsNode(
                    "count",
                    new I18nizableText("plugin.survey", "PLUGINS_SURVEY_USAGESTATISTICS_COUNT_LABEL"),
                    "ametysicon-maths-number-zero-one",
                    surveyIds.size(),
                    List.of(
                        new StatisticsValue(
                            "count",
                            new I18nizableText("plugin.survey", "PLUGINS_SURVEY_USAGESTATISTICS_COUNT_WITH_A_VALUE_LABEL"),
                            "ametysicon-maths-number-zero-one",
                            counts.size()
                        )
                    ),
                    true
                ),
                new StatisticsNode(
                    "answers",
                    new I18nizableText("plugin.survey", "PLUGINS_SURVEY_USAGESTATISTICS_ANSWERS_LABEL"),
                    "ametysicon-art-pencil",
                    count,
                    List.of(
                        new StatisticsValue(
                            "max",
                            new I18nizableText("plugin.survey", "PLUGINS_SURVEY_USAGESTATISTICS_ANSWERS_MAX_LABEL"),
                            "ametysicon-sort51",
                            counts.size() > 0 ? counts.get(counts.size() - 1) : 0
                        ),
                        new StatisticsValue(
                            "median",
                            new I18nizableText("plugin.survey", "PLUGINS_SURVEY_USAGESTATISTICS_ANSWERS_MEDIAN_LABEL"),
                            "ametysicon-maths-window-symbol-x",
                            counts.size() > 0 ? counts.get(counts.size() / 2) : 0
                        )                        
                    ),
                    false
                )
            ),
            true
        );
    }
    
    private List<String> _getSurveyIds()
    {
        List<String> surveyIds = new ArrayList<>();
        try (AmetysObjectIterable<AmetysObject> nodes = _ametysObjectResolver.query("//element(*, ametys:survey)"))
        {
            for (AmetysObject ametysObject : nodes)
            {
                Survey survey = (Survey) ametysObject;
                surveyIds.add(survey.getId());
            }
        }
        return surveyIds;
    }
}
