/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.tarteaucitron;

import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;
import org.ametys.runtime.plugins.admin.statistics.Statistics;
import org.ametys.runtime.plugins.admin.statistics.StatisticsNode;
import org.ametys.runtime.plugins.admin.statistics.StatisticsProvider;
import org.ametys.runtime.plugins.admin.statistics.StatisticsValue;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * All statistics of TarteAuCitron
 */
public class TarteAuCitronStatisticsProvider implements StatisticsProvider, Serviceable, PluginAware
{
    private SiteManager _siteManager;

    private String _id;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }
    
    public Statistics getStatistics()
    {
        return new StatisticsNode(
                _id,
                new I18nizableText("plugin.tarteaucitron", "PLUGINS_TARTEAUCITRON_STATISTICS_TARTEAUCITRON_LABEL"),
                "ametysicon-desktop-clipboard-list",
                null,
                List.of(
                    _getPluginsStatisticsOverral(),
                    _getPluginsStatisticsBySite()
                ),
                true
            );
    }

    
    private Statistics _getPluginsStatisticsOverral()
    {
        return new StatisticsValue(
            "tcservices",
            new I18nizableText("plugin.tarteaucitron", "PLUGINS_TARTEAUCITRON_STATISTICS_TARTEAUCITRON_GLOBAL_OVERRIDE_LABEL"),
            "ametysicon-object-broom",
            Config.getInstance().getValue("plugin.tarteaucitron.gdpr.param.override", false, false)
        );
    }
    
    private Statistics _getPluginsStatisticsBySite()
    {
        try (AmetysObjectIterable<Site> sites = _siteManager.getSites())
        {
            
            Long sitesUsingOverride = sites.stream()
                    .filter(site -> site.getValue("tarteaucitron-override", false, false))
                    .count();

            return new StatisticsValue(
                        "overrideBySites",
                        new I18nizableText("plugin.tarteaucitron", "PLUGINS_TARTEAUCITRON_STATISTICS_TARTEAUCITRON_GLOBAL_OVERRIDE_SITE"),
                        "ametysicon-object-broom",
                        sitesUsingOverride);
        }
        catch (Exception e)
        {
            throw new IllegalStateException(e);
        }
    }
    
}
