/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.tarteaucitron;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import org.ametys.core.util.dom.AmetysNodeList;
import org.ametys.core.util.dom.StringElement;
import org.ametys.runtime.config.Config;
import org.ametys.web.gdpr.GDPRService;
import org.ametys.web.gdpr.GDPRServicesExtensionPoint;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Helper component to be used from XSL stylesheets.
 */
public class TarteAuCitronXSLTHelper implements Serviceable
{
    /** The GDPR services extension point */
    protected static GDPRServicesExtensionPoint _gdprServicesEP;

    /** The site manager */
    protected static SiteManager _siteManager;
    
    // Mapping between Ametys GDPR identifier and Tarteaucitron identifier
    private static final Map<String, String> _TARTEAUCITRON_MAPPING = Map.of(
        "youtube", "youtube",
        "google-analytics", "gajs",
        "dailymotion", "dailymotion",
        "sharethis", "sharethis",
        "google-maps", "googlemaps",
        "google-calendar", "gagenda",
        "youtube-playlist", "youtubeplaylist",
        "matomo", "matomocloud"
    );
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _gdprServicesEP = (GDPRServicesExtensionPoint) manager.lookup(GDPRServicesExtensionPoint.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    /**
     * Returns the list of active service ids
     * @param siteName the site name
     * @return the list of active service ids
     */
    public static NodeList getServiceIds(String siteName)
    {
        List<Node> services = new ArrayList<>();
        
        Site site = _siteManager.getSite(siteName);
        List<String> serviceIds = _getConfiguredServicesIds(site);
        if (!_isConfigurationOverridden(site))
        {
            serviceIds.addAll(_getDefaultServiceIds(site));
        }
        
        for (String serviceId : serviceIds)
        {
            services.add(new StringElement("service", _TARTEAUCITRON_MAPPING.getOrDefault(serviceId, serviceId)));
        }
        
        return new AmetysNodeList(services);
    }

    /**
     * Get the list of default TarteAuCitron service ids
     * @param site the site
     * @return the list of default TarteAuCitron service ids
     */
    protected static List<String> _getDefaultServiceIds(Site site)
    {
        List<String> serviceIds = new ArrayList<>();
        
        List<GDPRService> gdprServices = _gdprServicesEP.getExtensionsIds()
                .stream()
                .map(id -> _gdprServicesEP.getExtension(id))
                .collect(Collectors.toList());
            
        List<GDPRService> tarteAuCitronServices = gdprServices.stream()
            .filter(service -> service.getGDPRId().isPresent() && "gdpr.tarteaucitron".equals(service.getGDPRId().get())) //filter only on tarteaucitron GDPR services
            .collect(Collectors.toList());
        
        Optional<String> googleAnalyticsId = Optional.ofNullable(site)
            .map(s -> s.<String>getValue("google-web-property-id"))
            .filter(StringUtils::isNotBlank);
        
        Optional<String> googleMapsId = Optional.ofNullable(site)
            .map(s -> s.<String>getValue("google-api-key"))
            .filter(StringUtils::isNotBlank);
        
        for (GDPRService service : tarteAuCitronServices)
        {
            String serviceId = service.getId();
            if ("google-analytics".equals(serviceId) && googleAnalyticsId.isPresent()
                || "google-maps".equals(serviceId) && googleMapsId.isPresent()
                || "youtube-playlist".equals(serviceId) && googleMapsId.isPresent() && _hasServiceEmptyGDPR(gdprServices, serviceId)
                || _hasServiceEmptyGDPR(gdprServices, serviceId))
            {
                serviceIds.add(_TARTEAUCITRON_MAPPING.getOrDefault(serviceId, serviceId));
            }
        }
        
        return serviceIds;
    }
    
    private static boolean _hasServiceEmptyGDPR(List<GDPRService> gdprServices, String serviceId)
    {
        return gdprServices.stream()
            .filter(service -> service.getId().equals(serviceId) && service.getGDPRId().isEmpty()) //test if the serviceId has a default implementation (with no GDPR)
            .findAny()
            .isPresent();
    }
    
    /**
     * <code>true</code> if the default configuration is overridden
     * @param site the site
     * @return <code>true</code> if the default configuration is overridden
     */
    protected static boolean _isConfigurationOverridden(Site site)
    {
        String siteServices = site.getValue("tarteaucitron-services");
        if (StringUtils.isNotBlank(siteServices))
        {
            return site.getValue("tarteaucitron-override", false, false);
        }
        
        String services = Config.getInstance().getValue("plugin.tarteaucitron.gdpr.param.services");
        return StringUtils.isNotBlank(services) && Config.getInstance().getValue("plugin.tarteaucitron.gdpr.param.override", false, false);
    }
    
    /**
     * Get the list of configured TarteAuCitron service ids
     * @param site the site
     * @return the list of service ids
     */
    protected static List<String> _getConfiguredServicesIds(Site site)
    {
        String siteServices = site.getValue("tarteaucitron-services");
        String services = StringUtils.isNotBlank(siteServices) ? siteServices : Config.getInstance().getValue("plugin.tarteaucitron.gdpr.param.services");
        if (StringUtils.isNotBlank(services))
        {
            return Stream.of(StringUtils.split(services, ","))
                .filter(StringUtils::isNotBlank)
                .map(String::trim)
                .collect(Collectors.toList());
        }
        
        return new ArrayList<>();
    }
}
