/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.translationflagging;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;

/**
 * Translation page DAO
 */
public class TranslationPageDAO extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = TranslationPageDAO.class.getName();

    private AmetysObjectResolver _resolver;
    private CurrentUserProvider _currentUserProvider;
    private ObservationManager _observationManager;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
    }
    
    /**
     * Set the translations for a specific page
     * @param pageId The current page id
     * @param pages the map of translated page
     * @return The list of updated pages.
     */
    public List<String> setTranslations(String pageId, Map<String, Page> pages)
    {
        Page currentPage = _resolver.resolveById(pageId);
        
        // Get the collection of pages to clean.
        Set<Page> pagesToClean = new HashSet<>(getTranslations(currentPage).values());
        
        // Compute all modified pages to notify the observer.
        Set<Page> modifiedPages = new HashSet<>(pagesToClean);
        modifiedPages.addAll(pages.values());
        
        // The old pages minus the new ones will be the pages to clean.
        pagesToClean.removeAll(pages.values());
        
        // Clean the old pages: remove the translations metadata.
        for (Page page : pagesToClean)
        {
            if (page instanceof ModifiablePage)
            {
                _cleanPage((ModifiablePage) page);
            }
        }
        
        // Set the translations.
        for (Page page : pages.values())
        {
            if (page instanceof ModifiablePage)
            {
                _setTranslations((ModifiablePage) page, pages);
            }
        }
        
        // Notify observers that page data has been changed
        
        List<String> modifiedPageIds = new ArrayList<>();
        for (Page modifiedPage : modifiedPages)
        {
            if (modifiedPage != null)
            {
                modifiedPageIds.add(modifiedPage.getId());
                
                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put(ObservationConstants.ARGS_PAGE, modifiedPage);

                _observationManager.notify(new Event(ObservationConstants.EVENT_PAGE_UPDATED, _currentUserProvider.getUser(), eventParams));
            }
        }
        
        return modifiedPageIds;
    }
    
    /**
     * Get the translations of a given page.
     * @param page the page.
     * @return the translated pages as a Map of pages, indexed by sitemap name (language).
     */
    public Map<String, Page> getTranslations(Page page)
    {
        Map<String, Page> translations = new HashMap<>();
        
        ModelLessDataHolder translationsComposite = page.getComposite(TranslationFlaggingClientSideElement.TRANSLATIONS_META);
        
        if (translationsComposite != null)
        {
            for (String lang : translationsComposite.getDataNames())
            {
                String translatedPageId = translationsComposite.getValue(lang);
                
                try
                {
                    Page translatedPage = _resolver.resolveById(translatedPageId);
                    translations.put(lang, translatedPage);
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Ignore : a removed page should be ignored
                }
            }
        }
        else
        {
            // Ignore : the translations composite doesn't exist, just return an empty map.
        }        
        
        return translations;
    }
    
    /**
     * Set the translations of a page in its data.
     * @param page the page.
     * @param pages the translated pages to set.
     */
    protected void _setTranslations(ModifiablePage page, Map<String, Page> pages)
    {
        ModifiableModelLessDataHolder translationsComposite = page.getComposite(TranslationFlaggingClientSideElement.TRANSLATIONS_META, true);
        
        for (Entry<String, Page> entry : pages.entrySet())
        {
            if (!entry.getKey().equals(page.getSitemapName()))
            {
                if (entry.getValue() == null)
                {
                    // Remove data.
                    translationsComposite.removeValue(entry.getKey());
                }
                else
                {
                    translationsComposite.setValue(entry.getKey(), entry.getValue().getId());
                }
            }
        }
        
        page.saveChanges();
    }
    
    /**
     * Clean a page of its translations data.
     * @param page the page to clean.
     */
    protected void _cleanPage(ModifiablePage page)
    {
        page.removeValue(TranslationFlaggingClientSideElement.TRANSLATIONS_META);
        if (page.needsSave())
        {
            page.saveChanges();
        }
    }
}
