/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.translationflagging;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * This element creates a ribbon button to flag translation on a page.
 * Provides callable methods to set and get translations.
 */
public class TranslationFlaggingClientSideElement extends StaticClientSideElement
{
    /** Constant for the translations metadata name */
    public static final String TRANSLATIONS_META = "translations";
    
    private AmetysObjectResolver _resolver;
    private SiteManager _siteManager;
    private TranslationPageDAO _translationPageDAO;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _translationPageDAO = (TranslationPageDAO) smanager.lookup(TranslationPageDAO.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        String siteName = (String) contextParameters.get("siteName");
        if (siteName != null)
        {
            Site site = _siteManager.getSite(siteName);
            if (site.getSitemaps().getSize() > 1)
            {
                return super.getScripts(ignoreRights, contextParameters);
            }
        }
        
        return new ArrayList<>();
    }
    
    /**
     * Get all the translations associated with a page
     * @param pageId The page Id
     * @return The page and its translations data
     */
    @Callable
    public Map<String, Object> getTranslations(String pageId)
    {
        Map<String, Object> result = new HashMap<>();

        Page page = _resolver.resolveById(pageId);
        result.put("page", _page2json(page));
        
        Map<String, Page> translatedPages = _translationPageDAO.getTranslations(page);
        
        List<Map<String, String>> translations = new ArrayList<>();
        for (Page translatedPage : translatedPages.values())
        {
            translations.add(_page2json(translatedPage));
        }
        result.put("translations", translations);
        
        return result;
    }
    
    private Map<String, String> _page2json (Page page)
    {
        Map<String, String> pageData = new HashMap<>();
        
        pageData.put("id", page.getId());
        pageData.put("site", page.getSiteName());
        pageData.put("lang", page.getSitemapName());
        pageData.put("path", page.getPathInSitemap());
        pageData.put("title", page.getTitle());
        
        return pageData;
    }
    
    /**
     * Set the translations for a specific page
     * @param pageId The current page id
     * @param translations an association language-pageId, to set as translations
     * @return The list of updated pages.
     */
    @Callable
    public List<String> setTranslations(String pageId, Map<String, String> translations)
    {
        // Get the translated pages from the request.
        Map<String, Page> pages = _getTranslatedPages(translations, pageId);

        return _translationPageDAO.setTranslations(pageId, pages);
    }
    
    /**
     * Get translated pages from the map.
     * @param translations the associative list of translations.
     * @param pageId the current page ID.
     * @return the translated pages as a Map of pages, indexed by sitemap name.
     */
    protected Map<String, Page> _getTranslatedPages(Map<String, String> translations, String pageId)
    {
        Map<String, Page> pages = new HashMap<>();
        
        Page currentPage = _resolver.resolveById(pageId);
        String siteName = currentPage.getSiteName();
        
        pages.put(currentPage.getSitemapName(), currentPage);
        
        Site site = _siteManager.getSite(siteName);
        
        AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps();
        
        for (Sitemap sitemap : sitemaps)
        {
            String name = sitemap.getSitemapName();
            if (!name.equals(currentPage.getSitemapName()))
            {
                if (translations.containsKey(name) && StringUtils.isNotBlank(translations.get(name)))
                {
                    Page page = _resolver.resolveById(translations.get(name));
                    if (name.equals(page.getSitemapName()))
                    {
                        pages.put(name, page);
                    }
                }
                else
                {
                    pages.put(name, null);
                }
            }
        }
        
        return pages;
    }
}
