/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.actions;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.captcha.CaptchaHelper;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.ugc.UGCConstants;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.content.FOContentCreationHelper;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.site.Site;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

import jakarta.mail.MessagingException;

/**
 * Create content action
 */
public class ProposeContentAction extends ServiceableAction
{
    private static final int _INITIAL_ACTION_ID = 1111;
    
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** The content type helper */
    protected ContentTypesHelper _contentTypeHelper;
    
    /** The JSON Utils */
    protected JSONUtils _jsonUtils;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The I18n utils */
    protected I18nUtils _i18nUtils;

    /** The user manager */
    protected UserManager _userManager;

    /** Helper for FO content creation */
    protected FOContentCreationHelper _foContentCreationHelper;
    
    /** The page helper */
    protected PageHelper _pageHelper;

    /** The content helper */
    protected ContentHelper _contentHelper;
    /** The UserLanguagesManager */
    protected UserLanguagesManager _userLanguagesManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _contentTypeHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _i18nUtils = (I18nUtils) serviceManager.lookup(I18nUtils.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _foContentCreationHelper = (FOContentCreationHelper) serviceManager.lookup(FOContentCreationHelper.ROLE);
        _pageHelper = (PageHelper) serviceManager.lookup(PageHelper.ROLE);
        _contentHelper = (ContentHelper) serviceManager.lookup(ContentHelper.ROLE);
        _userLanguagesManager = (UserLanguagesManager) serviceManager.lookup(UserLanguagesManager.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        Multimap<String, I18nizableText> errors = ArrayListMultimap.create();

        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String zoneId = request.getParameter("zoneId");
        ZoneItem zoneItem = _resolver.resolveById(zoneId);
        Page page = zoneItem.getParent().getParent().getParent().getParent();
        
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        String workflowName = serviceParameters.getValue("workflow");
        String contentTypeId = serviceParameters.getValue("content-type");
        UserIdentity[] adminUsers = serviceParameters.getValue("users");
        
        String siteName = request.getParameter("site");
        String language = request.getParameter("lang");
        
        ContentType ugcMixin = _cTypeEP.getExtension(UGCConstants.UGC_MIXIN_TYPE);
        
        ContentType contentType = _cTypeEP.getExtension(contentTypeId);
        Map<String, Object> values = _foContentCreationHelper.getAndValidateFormValues(request, contentType, "main", errors);
        values.putAll(_foContentCreationHelper.getAndValidateFormValues(request, ugcMixin, "main", errors));
        
        // Check captcha if needed
        if (_pageHelper.isCaptchaRequired(page))
        {
            String captchaValue = request.getParameter("captcha");
            String captchaKey = request.getParameter("captcha-key");

            // Validate the input
            if (!CaptchaHelper.checkAndInvalidate(captchaKey, captchaValue))
            {
                errors.put("captcha", new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_FORM_ERROR_INVALID_CAPTCHA"));
            }
        }
        
        String gtuMode = zoneItem.getServiceParameters().getValue("general-terms-of-use-mode", false, "NONE");
        if (!"NONE".equals(gtuMode) && !"true".equals(request.getParameter("gtu")))
        {
            errors.put("gtu", new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_FORM_ERROR_INVALID_GTU"));
        }
        
        // If there are no errors
        if (errors.isEmpty())
        {
            String title = request.getParameter("title");
            
            // we add a prefix to the title for the content name to prevent issue with numeric title
            String prefix = StringUtils.substringAfterLast(contentTypeId, ".");
            
            result = _foContentCreationHelper.createAndEditContent(_INITIAL_ACTION_ID, new String[] {contentTypeId}, new String[] {UGCConstants.UGC_MIXIN_TYPE}, siteName, prefix + "-" + title, title, language, values, workflowName, null, new HashMap<>());
            
            if (result.containsKey(Content.class.getName()))
            {
                result.put("success", true);

                // Send mail
                WorkflowAwareContent content = (WorkflowAwareContent) result.get(Content.class.getName());
                _sendMail(adminUsers, content);
            }
            else
            {
                result.put("success", false);
            }
        }
        else
        {
            result.put("success", false);
            result.put("errors", _jsonUtils.convertObjectToJson(errors.asMap()));
        }
        
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        return EMPTY_MAP;
    }

    /**
     * Send mail to inform that a content is created
     * @param adminUsers the admin users
     * @param content the created content
     */
    protected void _sendMail(UserIdentity[] adminUsers, WorkflowAwareContent content)
    {
        if (adminUsers != null)
        {
            String creatorFullName = null;
            UserIdentity creatorIdentity = _currentUserProvider.getUser();
            if (creatorIdentity != null)
            {
                User creator = _userManager.getUser(creatorIdentity);
                creatorFullName = creator.getFullName();
            }
            else
            {
                creatorFullName = content.getValue(UGCConstants.ATTRIBUTE_PATH_UGC_AUTHOR);
            }
            
            Site site = _getSite(content);
            
            I18nizableText i18nSubject = _getI18nSubject(site);
            MailBodyBuilder bodyBuilder = _getI18nBody(content, creatorFullName, site);
            
            Map<String, MailContent> mailContentByLanguage = new HashMap<>();
            
            for (UserIdentity adminUser : adminUsers)
            {
                User user = _userManager.getUser(adminUser);
                if (user == null)
                {
                    getLogger().error("Can not send an e-mail for the content creation to the unexisting user " + adminUser);
                }
                else
                {
                    String userLanguage = StringUtils.defaultIfBlank(user.getLanguage(), _userLanguagesManager.getDefaultLanguage());

                    String subject = null;
                    String htmlBody = null;
                    
                    MailContent mailContent = mailContentByLanguage.computeIfAbsent(userLanguage, lang -> {
                        try
                        {
                            return new MailContent(_i18nUtils.translate(i18nSubject, lang), bodyBuilder.withLanguage(lang).build());
                        }
                        catch (IOException e)
                        {
                            getLogger().warn("Fail to build HTML body for the content creation", e);
                        }
                        
                        return null;
                    });
                    
                    if (mailContent != null)
                    {
                        subject = mailContent.subject();
                        htmlBody = mailContent.body();
                        
                        String email = user.getEmail();
                        try
                        {
                            SendMailHelper.newMail()
                                .withSubject(subject)
                                .withHTMLBody(htmlBody)
                                .withRecipient(email)
                                .sendMail();
                        }
                        catch (MessagingException | IOException e)
                        {
                            getLogger().warn("Could not send an e-mail for the content creation to " + email, e);
                        }
                    }
                }
            }
        }
    }

    /**
     * Get the i18n body
     * @param content the content
     * @param creatorFullName the creator full name
     * @param site the site
     * @return The i18n body
     */
    protected MailBodyBuilder _getI18nBody(WorkflowAwareContent content, String creatorFullName, Site site)
    {
        List<String> parametersBody = new ArrayList<>();
        parametersBody.add(content.getTitle());
        parametersBody.add(creatorFullName);
        
        MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
            .withTitle(_getI18nSubject(site));
            
        if (site != null)
        {
            parametersBody.add(_getContentUri(content));
            parametersBody.add(site.getTitle());
            parametersBody.add(site.getUrl());
            
            bodyBuilder.withMessage(new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_FORM_CREATED_CONTENT_MAIL_BODY", parametersBody))
                .withLink(_getContentUri(content), new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_FORM_CREATED_CONTENT_MAIL_BODY_CONTENT_LINK"));
        }
        else
        {
            bodyBuilder.withMessage(new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_FORM_CREATED_CONTENT_MAIL_BODY_NO_SITE", parametersBody));
        }
        return bodyBuilder;
    }

    /**
     * Get the i18n subject
     * @param site the site
     * @return The i18n subject
     */
    protected I18nizableText _getI18nSubject(Site site)
    {
        I18nizableText i18nSubject = null;
        if (site != null)
        {
            List<String> parametersSubject = new ArrayList<>();
            parametersSubject.add(site.getTitle());
            i18nSubject = new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_FORM_CREATED_CONTENT_MAIL_SUBJECT", parametersSubject);
        }
        else
        {
            i18nSubject = new I18nizableText("plugin.ugc", "PLUGINS_UGC_SERVICE_FORM_CREATED_CONTENT_MAIL_SUBJECT_NO_SITE");
        }
        return i18nSubject;
    }
    
    /**
     * Get the site name
     * @param content The content
     * @return the site name
     */
    protected Site _getSite(WorkflowAwareContent content)
    {
        if (content instanceof WebContent)
        {
            return ((WebContent) content).getSite();
        }
        
        return null;
    }
    
    
    /**
     * Get the content uri
     * @param content the content
     * @return the content uri
     */
    protected String _getContentUri(WorkflowAwareContent content)
    {
        return _contentHelper.getContentBOUrl(content, Map.of());
    }
    
    private record MailContent(String subject, String body) { /* Nothing */ }
}
