/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.ugc.activities;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.ugc.UGCConstants;
import org.ametys.plugins.ugc.page.UGCPage;
import org.ametys.plugins.ugc.page.UGCPageHandler;
import org.ametys.web.activities.PageUpdatedActivityType;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * {@link ActivityType} implementation for UGC content validated activities
 */
public class UGCPageUpdatedActivityType extends PageUpdatedActivityType
{
    /** The content types helper */
    protected ContentTypesHelper _contentTypesHelper;
    
    /** The UGC page handler */
    protected UGCPageHandler _ugcPageHandler;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _contentTypesHelper = (ContentTypesHelper) serviceManager.lookup(ContentTypesHelper.ROLE);
        _ugcPageHandler = (UGCPageHandler) serviceManager.lookup(UGCPageHandler.ROLE);
    }
    
    @Override
    protected List<Page> getPages(Event event)
    {
        Map<String, Object> args = event.getArguments();
        String contentId = (String) args.get(ObservationConstants.ARGS_CONTENT_ID);
        Content content = _resolver.resolveById(contentId);

        List<Page> pages = new ArrayList<>();
        if (_contentTypesHelper.isInstanceOf(content, UGCConstants.UGC_MIXIN_TYPE))
        {
            for (Site site : _siteManager.getSites())
            {
                for (String type : content.getTypes())
                {
                    Optional<UGCPage> ugcPage = _ugcPageHandler.getUgcPage(content, site.getName(), content.getLanguage(), type);
                    if (ugcPage.isPresent())
                    {
                        pages.add(ugcPage.get());
                    }
                }
            }
        }
        
        return pages;
    }
}
