/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.MetadataExpression;
import org.ametys.plugins.repository.query.expression.NotExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.plugins.userdirectory.page.VirtualOrganisationChartPageFactory;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;

/**
 * Component providing methods to retrieve organization chart virtual pages, such as the organization chart root and orgUnit page.
 */
public class OrganisationChartPageHandler extends AbstractLogEnabled implements Component, Serviceable
{
    /** The avalon role. */
    public static final String ROLE = OrganisationChartPageHandler.class.getName();
    
    /** The orgUnit parent attribute name */
    public static final String PARENT_ORGUNIT_ATTRIBUTE_NAME = "parentOrgUnit";
    
    /** The orgUnit child attribute name */
    public static final String CHILD_ORGUNIT_ATTRIBUTE_NAME = "childOrgUnits";
    
    /** The attribute name for orgUnit users repeater */
    public static final String ORGUNIT_USERS_ATTRIBUTE_NAME = "users";
    
    /** The attribute name for orgUnit user in repeater */
    public static final String ORGUNIT_USER_ATTRIBUTE_NAME = "user";
    
    /** The attribute name for orgUnit user role in repeater */
    public static final String ORGUNIT_USER_ROLE_ATTRIBUTE_NAME = "role";

    /** The data name for the content type of the orgUnit chart */
    public static final String CONTENT_TYPE_DATA_NAME = "organisation-chart-root-contenttype";
    /** The data name for the child visiblity of the orgUnit chart */
    public static final String PAGE_VISIBLE_DATA_NAME = "organization-chart-page-visible";
    
    private static final Comparator<Content> __ORGUNIT_COMPARATOR = Comparator
        // Compare by order data
        .<Content>comparingLong(c -> c.<Long>getValue("order", false, Long.MAX_VALUE))
        // Then by title in lower case
        .thenComparing(c -> c.getTitle().toLowerCase(), Comparator.naturalOrder());
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get orgUnit contents from rootPage
     * @param rootPage the root page
     * @return the list of orgUnit contents
     */
    public Stream<Content> getContentsForRootPage(Page rootPage)
    {
        String lang = rootPage.getSitemapName();
        String contentType = getContentTypeId(rootPage);
        return getFirstLevelOfContents(lang, contentType);
    }
    
    /**
     * Get orgUnit contents
     * @param lang the language
     * @param contentType the content type of organization chart
     * @return the list of orgUnit contents at the first level
     */
    public Stream<Content> getFirstLevelOfContents(String lang, String contentType)
    {
        ContentTypeExpression contentTypeExp = new ContentTypeExpression(Operator.EQ, contentType);
        MetadataExpression parentMetadataExpression = new MetadataExpression(PARENT_ORGUNIT_ATTRIBUTE_NAME);
        OrExpression noParentExpression = new OrExpression(new NotExpression(parentMetadataExpression), new StringExpression(PARENT_ORGUNIT_ATTRIBUTE_NAME, Operator.EQ, ""));
        
        Expression finalExpr = new AndExpression(noParentExpression, contentTypeExp, new LanguageExpression(Operator.EQ, lang));
        
        SortCriteria sort = new SortCriteria();
        sort.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
        
        String xPathQuery = QueryHelper.getXPathQuery(null, "ametys:content", finalExpr, sort);
        
        return _resolver.<Content>query(xPathQuery).stream().sorted(__ORGUNIT_COMPARATOR);
    }
    
    /**
     * True if the page is the organization chart root page
     * @param jcrPage the page
     * @return true if the page is the organization chart root page
     */
    public boolean isOrganisationChartRootPage (JCRAmetysObject jcrPage)
    {
        try
        {
            Node node = jcrPage.getNode();
            
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                List<Value> values = Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues());
                
                return values.stream()
                        .map(LambdaUtils.wrap(Value::getString))
                        .anyMatch(v -> VirtualOrganisationChartPageFactory.class.getName().equals(v));
            }
            else
            {
                return false;
            }
        }
        catch (RepositoryException e)
        {
            return false;
        }
    }
    
    /**
     * Get child orgUnit contents from parentContent
     * @param parentContent the parent content
     * @return the list of child orgUnit contents
     */
    public List<Content> getChildContents(Content parentContent)
    {
        List<Content> contentList = new ArrayList<>();

        ContentValue[] contents = parentContent.getValue(CHILD_ORGUNIT_ATTRIBUTE_NAME);
        if (contents != null)
        {
            for (ContentValue content : contents)
            {
                try
                {
                    contentList.add(content.getContent());
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().warn("The parent entity {} ({}) is referencing an unexisting child node '{}'", parentContent.getTitle(), parentContent.getId(), content.getContentId(), e);
                }
            }
        }
        
        contentList.sort(__ORGUNIT_COMPARATOR);
        
        return contentList;
    }
    
    /**
     * Get parent orgUnit content from childContent
     * @param childContent the child content
     * @return the parent orgUnit content
     */
    public Content getParentContent(Content childContent)
    {
        ContentValue content = childContent.getValue(PARENT_ORGUNIT_ATTRIBUTE_NAME);
        if (content != null)
        {
            try
            {
                return content.getContent();
            }
            catch (UnknownAmetysObjectException e)
            {
                getLogger().warn("There is no parent content with id " + content.getContentId(), e);
            }
        }
        
        return null;
    }
    
    /**
     * Get users contents from content
     * @param content the content
     * @return the list of user contents
     */
    public List<Content> getUserContents(Content content)
    {
        List<Content> users = new ArrayList<>();
        
        ContentValue[] contents = content.getValue(ORGUNIT_USERS_ATTRIBUTE_NAME + "/" + ORGUNIT_USER_ATTRIBUTE_NAME, true);
        
        if (contents != null)
        {
            for (ContentValue contentValue : contents)
            {
                try
                {
                    users.add(contentValue.getContent());
                }
                catch (Exception e)
                {
                    getLogger().warn("The entity {} ({}) is referencing an unexisting user content node '{}'", content.getTitle(), content.getId(), contentValue.getContentId(), e);
                }
            }
        }
        
        return users;
    }
    
    /**
     * Get the child content. Return null if it not exist
     * @param parentContent the parent content
     * @param path the path from the parent content
     * @return the child content.
     */
    public Content getChildFromPath(Content parentContent, String path)
    {
        String contentName = path.contains("/") ? StringUtils.substringBefore(path, "/") : path;
        
        List<Content> childContents = getChildContents(parentContent);
        List<Content> contentFilter = childContents.stream().filter(c -> c.getName().equals(contentName)).collect(Collectors.toList());
        
        if (!contentFilter.isEmpty())
        {
            if (path.contains("/"))
            {
                return getChildFromPath(contentFilter.get(0), StringUtils.substringAfter(path, "/"));
            }
            else
            {
                return contentFilter.get(0);
            }
        }
        
        return null;
    }
    
    /**
     * Get the organization chart root page.
     * @param siteName the current site.
     * @param sitemapName the sitemap name.
     * @return the organization chart root page
     * @throws AmetysRepositoryException if an error occurred.
     */
    public Set<Page> getOrganisationChartRootPages(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Expression expression = new VirtualFactoryExpression(VirtualOrganisationChartPageFactory.class.getName());
        
        String query = PageQueryHelper.getPageXPathQuery(siteName, sitemapName, null, expression, null);
        
        AmetysObjectIterable<Page> pages = _resolver.query(query);

        return pages.stream().collect(Collectors.toSet());
    }
    
    /**
     * Get the organization chart root page.
     * @param siteName the current site.
     * @param sitemapName the sitemap name.
     * @param contentTypeId The content type id
     * @return the organization chart root page
     * @throws AmetysRepositoryException if an error occurred.
     */
    public Page getOrganisationChartRootPage(String siteName, String sitemapName, String contentTypeId) throws AmetysRepositoryException
    {
        String contentTypeIdToCompare = contentTypeId != null ? contentTypeId : "";
        
        for (Page orgUnitRootPage : getOrganisationChartRootPages(siteName, sitemapName))
        {
            if (contentTypeIdToCompare.equals(getContentTypeId(orgUnitRootPage)))
            {
                return orgUnitRootPage;
            }
        }
        
        return null;
    }
    
    /**
     * Gets the content type id
     * @param rootPage The organisation chart root page
     * @return the content type id
     */
    public String getContentTypeId(Page rootPage)
    {
        return rootPage.getValue(CONTENT_TYPE_DATA_NAME);
    }
}
