/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.forms;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.forms.question.computing.AbstractStaticComputingType;
import org.ametys.plugins.forms.repository.FormEntry;
import org.ametys.plugins.forms.repository.FormQuestion;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.type.ModelItemType;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Class for creating user directory computed field
 */
public class UDUserComputingQuestionType extends AbstractStaticComputingType
{
    /** The accepted types for content attributes */
    private static final List<String> __ACCEPTED_TYPES = List.of(
            ModelItemTypeConstants.BOOLEAN_TYPE_ID,
            ModelItemTypeConstants.DATE_TYPE_ID,
            ModelItemTypeConstants.DATETIME_TYPE_ID,
            ModelItemTypeConstants.DOUBLE_TYPE_ID,
            ModelItemTypeConstants.LONG_TYPE_ID,
            ModelItemTypeConstants.STRING_TYPE_ID
            );
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The form question ud helper */
    protected FormQuestionUDHelper _formQuestionUDHelper;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** Map of ModelItems specific to UDUserComputingQuestionType */
    protected Map<String, ModelItem> _userComputingItems;
   
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _formQuestionUDHelper = (FormQuestionUDHelper) manager.lookup(FormQuestionUDHelper.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        
    }
    
    public Object getComputedValue(FormQuestion computedQuestion, FormEntry entry)
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        return userIdentity != null ? _formQuestionUDHelper.getUserContentAttributeValue(computedQuestion, userIdentity) : null;
    }

    public Map<String, ModelItem> getModelItems()
    {
        _userComputingItems = _formQuestionUDHelper.getUDModelItems(__ACCEPTED_TYPES);
        return _userComputingItems;
    }

    public List<ViewItem> getViewElements()
    {
        List<ViewItem> viewElements = new LinkedList<>();
        
        for (Entry<String, ModelItem> modelItem : _userComputingItems.entrySet())
        {
            ViewElement userSelect = new ViewElement();
            userSelect.setDefinition((ElementDefinition< ? >) modelItem.getValue());
            viewElements.add(userSelect);
        }
        
        return viewElements;
    }

    public String getStorageType(FormQuestion question)
    {
        ModelItemType type = _getAttributeType(question);
        return type.getId();
    }

    private ModelItemType _getAttributeType(FormQuestion question)
    {
        String contentTypeId = question.getValue(FormQuestionUDHelper.ATTRIBUTE_USER_CONTENT_TYPE);
        ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
        String userAttribute = question.getValue(FormQuestionUDHelper.ATTRIBUTE_UD_USER_ATTRIBUTE_NAME + "-" + contentTypeId.replace(".", "-"));
        ModelItem modelItem = contentType.getModelItem(userAttribute);
        return modelItem.getType();
    }
    
    @Override
    public List<String> getFieldToDisableIfFormPublished()
    {
        List<String> fieldNames =  super.getFieldToDisableIfFormPublished();
        for (String attributeId : _userComputingItems.keySet())
        {
            fieldNames.add(attributeId);
        }
        return fieldNames;
    }
    
    public boolean canEdit()
    {
        return false;
    }
    
    @Override
    public void saxAdditionalValue(ContentHandler contentHandler, FormQuestion question, FormEntry entry) throws SAXException
    {
        ModelItemType type = _getAttributeType(question);
        XMLUtils.createElement(contentHandler, "computing-attribut-type", type.getId());
    }
    
    public Map<String, Object> getAnonymizedData(FormQuestion question)
    {
        return Map.of(question.getNameForForm(), "content://anonymized");
    }
}
