/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.synchronize;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.directory.StoredUser;
import org.ametys.core.user.directory.UserDirectory;
import org.ametys.core.user.population.UserPopulation;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.contentio.synchronize.AbstractSimpleSynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.impl.LDAPCollectionHelper;
import org.ametys.plugins.contentio.synchronize.impl.LDAPCollectionHelper.LDAPCollectionHelperSearchResult;
import org.ametys.plugins.core.impl.user.directory.JdbcUserDirectory;
import org.ametys.plugins.core.impl.user.directory.LdapUserDirectory;
import org.ametys.plugins.userdirectory.DeleteOrgUnitComponent;
import org.ametys.plugins.userdirectory.DeleteUserComponent;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Implementation of {@link SynchronizableContentsCollection} to be synchronized with a {@link UserPopulation} of the CMS.
 */
public class UserPopulationSynchronizableContentsCollection extends AbstractSimpleSynchronizableContentsCollection
{
    /** Name of parameter holding the id of population */
    protected static final String __PARAM_POPULATION_ID = "populationId";
    /** Name of parameter for the firstname attribute */
    protected static final String __PARAM_FIRSTNAME_ATTRIBUTE_NAME = "firstname";
    /** Name of parameter for the lastname attribute */
    protected static final String __PARAM_LASTNAME_ATTRIBUTE_NAME = "lastname";
    /** Name of parameter for the email attribute */
    protected static final String __PARAM_EMAIL_ATTRIBUTE_NAME = "email";
    /** Name of parameter holding the fields mapping */
    protected static final String __PARAM_MAPPING = "mapping";
    /** Name of parameter holding the additional search filter */
    protected static final String __PARAM_ADDITIONAL_SEARCH_FILTER = "additionalSearchFilter";
    /** Name of parameter into mapping holding the synchronized property */
    protected static final String __PARAM_MAPPING_SYNCHRO = "synchro";
    /** Name of parameter into mapping holding the path of attribute */
    protected static final String __PARAM_MAPPING_ATTRIBUTE_REF = "metadata-ref";
    /** Name of parameter into mapping holding the remote attribute */
    protected static final String __PARAM_MAPPING_ATTRIBUTE_PREFIX = "attribute-";

    /** The logger */
    protected static final Logger _LOGGER = LoggerFactory.getLogger(UserPopulationSynchronizableContentsCollection.class);
    
    /** The DAO for user populations */
    protected UserPopulationDAO _userPopulationDAO;
    /** The service manager */
    protected ServiceManager _manager;
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    /** The delete user component */
    protected DeleteUserComponent _deleteUserComponent;
    
    /** Mapping of the attributes with source data */
    protected Map<String, Map<String, List<String>>> _mapping;
    /** Synchronized fields */
    protected Set<String> _syncFields;
    /** LDAP collection helper fields */
    protected LDAPCollectionHelper _ldapHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _userPopulationDAO = (UserPopulationDAO) manager.lookup(UserPopulationDAO.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _deleteUserComponent = (DeleteUserComponent) manager.lookup(DeleteUserComponent.ROLE);
        _ldapHelper = (LDAPCollectionHelper) manager.lookup(LDAPCollectionHelper.ROLE);
        _manager = manager;
    }
    
    public boolean handleRightAssignmentContext()
    {
        return false;
    }
    
    @Override
    protected void configureDataSource(Configuration configuration) throws ConfigurationException
    {
        _mapping = new HashMap<>();
        _syncFields = new HashSet<>();
        String mappingAsString = (String) getParameterValues().get(__PARAM_MAPPING);
        if (StringUtils.isNotEmpty(mappingAsString))
        {
            List<Object> mappingAsList = _jsonUtils.convertJsonToList(mappingAsString);
            for (Object object : mappingAsList)
            {
                @SuppressWarnings("unchecked")
                Map<String, Object> field = (Map<String, Object>) object;
                String attributeRef = (String) field.get(__PARAM_MAPPING_ATTRIBUTE_REF);
                
                String prefix = __PARAM_MAPPING_ATTRIBUTE_PREFIX;
                for (String prefixedUserDirectoryKey : _getUserDirectoryKeys(field, prefix))
                {
                    String userDirectoryKey = prefixedUserDirectoryKey.substring(prefix.length());
                    if (!_mapping.containsKey(userDirectoryKey))
                    {
                        _mapping.put(userDirectoryKey, new HashMap<>());
                    }
                    
                    String[] attributes = ((String) field.get(prefixedUserDirectoryKey)).split(",");
                    
                    Map<String, List<String>> userDirectoryMapping = _mapping.get(userDirectoryKey);
                    userDirectoryMapping.put(attributeRef, Arrays.asList(attributes));
                }

                boolean isSynchronized = field.containsKey(__PARAM_MAPPING_SYNCHRO) ? (Boolean) field.get(__PARAM_MAPPING_SYNCHRO) : false;
                if (isSynchronized)
                {
                    _syncFields.add(attributeRef);
                }
            }
        }
    }

    @Override
    protected void configureSearchModel()
    {
        _LOGGER.info("Currently, SCC '{}' cannot define a search model", UserPopulationSynchronizableContentsCollection.class.getName());
    }
    
    @Override
    protected List<ModifiableContent> _internalPopulate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        List<ModifiableContent> contents = new ArrayList<>();
        
        UserPopulation population = _userPopulationDAO.getUserPopulation(getPopulationId());
        
        for (UserDirectory userDirectory : population.getUserDirectories())
        {
            String labelOrId = userDirectory.getLabel().isEmpty() ? userDirectory.getId() : userDirectory.getLabel();
            progressionTracker.addContainerStep(userDirectory.getId(), new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_SCHEDULER_SYNCHRONIZE_COLLECTION_USER_DIRECTORY_STEP_LABEL", List.of(labelOrId)));
        }
        
        for (UserDirectory userDirectory : population.getUserDirectories())
        {
            Map<String, Object> searchParams = new HashMap<>();
            searchParams.put("userDirectory", userDirectory);
            contents.addAll(_importOrSynchronizeContents(searchParams, false, logger, progressionTracker.getStep(userDirectory.getId())));
        }
        
        return contents;
    }
    
    /**
     * Search contents from a LDAP user directory of the population.
     * To avoid code duplication and useless operations, we return a {@link Map}&lt;{@link String}, {@link Map}&lt;{@link String}, {@link Object}&gt;&gt;
     * if getRemoteValues is set to false and {@link Map}&lt;{@link String}, {@link Map}&lt;{@link String}, {@link List}&lt;{@link Object}&gt;&gt;&gt;
     * if remoteValues is true.
     * Without this operation, we have to duplicate the code of searchLDAP and _internalSearch methods.
     * @param userDirectory The LDAP user directory
     * @param searchParameters Parameters for the search
     * @param offset Begin of the search
     * @param limit Number of results
     * @param logger The logger
     * @param getRemoteValues if <code>true</code>, values are organized by the attribute mapping
     * @return Contents found in LDAP
     */
    @SuppressWarnings("unchecked")
    protected Map<String, Map<String, Object>> searchLDAP(LdapUserDirectory userDirectory, Map<String, Object> searchParameters, int offset, int limit, Logger logger, boolean getRemoteValues)
    {
        Map<String, Map<String, Object>> searchResults = new HashMap<>();
        
        Map<String, Object> ldapParameterValues = userDirectory.getParameterValues();
        String dataSourceId = (String) ldapParameterValues.get(LdapUserDirectory.PARAM_DATASOURCE_ID);
        String relativeDN = (String) ldapParameterValues.get(LdapUserDirectory.PARAM_USERS_RELATIVE_DN);
        String filter = (String) ldapParameterValues.get(LdapUserDirectory.PARAM_USERS_OBJECT_FILTER);
        String searchScope = (String) ldapParameterValues.get(LdapUserDirectory.PARAM_USERS_SEARCH_SCOPE);
        String loginAttr = (String) ldapParameterValues.get(LdapUserDirectory.PARAM_USERS_LOGIN_ATTRIBUTE);
        
        Map<String, List<String>> udMapping = _mapping.getOrDefault(userDirectory.getId(), new HashMap<>());
        udMapping.put(getLoginAttributeName(), List.of(loginAttr));
        
        // If first name attribute is set
        String firstNameAttribute = getFirstNameAttributeName();
        if (StringUtils.isNotBlank(firstNameAttribute))
        {
            String firstNameAttr = (String) ldapParameterValues.get(LdapUserDirectory.PARAM_USERS_FIRSTNAME_ATTRIBUTE);
            udMapping.put(firstNameAttribute, List.of(firstNameAttr));
        }
        
        // If last name attribute is set
        String lastNameAttribute = getLastNameAttributeName();
        if (StringUtils.isNotBlank(lastNameAttribute))
        {
            String lastNameAttr = (String) ldapParameterValues.get(LdapUserDirectory.PARAM_USERS_LASTNAME_ATTRIBUTE);
            udMapping.put(lastNameAttribute, List.of(lastNameAttr));
        }
        
        // If email attribute is set
        String emailAttribute = getEmailAttributeName();
        if (StringUtils.isNotBlank(emailAttribute))
        {
            String emailAttr = (String) ldapParameterValues.get(LdapUserDirectory.PARAM_USERS_EMAIL_ATTRIBUTE);
            udMapping.put(emailAttribute, List.of(emailAttr));
        }
        
        try
        {
            List<String> filters = new ArrayList<>();
            if (StringUtils.isNotEmpty(filter))
            {
                filters.add(filter);
            }
            
            if (searchParameters != null)
            {
                for (String parameterName : searchParameters.keySet())
                {
                    filters.add(parameterName + "=" + searchParameters.get(parameterName));
                }
            }
            
            String additionalSearchFilter = getAdditionalSearchFilter();
            if (StringUtils.isNotEmpty(additionalSearchFilter))
            {
                filters.add(additionalSearchFilter);
            }
            
            String filtersReduced = filters.stream()
                                           .filter(StringUtils::isNotEmpty)
                                           .map(s -> "(" + s + ")")
                                           .reduce("", (s1, s2) -> s1 + s2);
            if (!filtersReduced.isEmpty())
            {
                filtersReduced = "(&" + filtersReduced + ")";
            }
            
            LDAPCollectionHelperSearchResult results  = _ldapHelper.search(getId(), relativeDN, filtersReduced, searchScope, offset, limit, udMapping, loginAttr, logger, dataSourceId);
            searchResults = results.searchResults();
            
            if (getRemoteValues)
            {
                Map<String, Map<String, List<Object>>> organizedResults = _sccHelper.organizeRemoteValuesByAttribute(searchResults, udMapping);
                _ldapHelper.transformTypedAttributes(organizedResults, getContentType(), udMapping.keySet());
                searchResults = (Map<String, Map<String, Object>>) (Object) organizedResults;
            }
            
            _nbError = results.nbErrors();
            _hasGlobalError = results.hasGlobalError();
        }
        catch (Exception e)
        {
            throw new RuntimeException("An error occured when importing from LDAP UserDirectory", e);
        }
        
        return searchResults;
    }
    
    @Override
    protected Map<String, Map<String, Object>> internalSearch(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        return _internalSearch(searchParameters, offset, limit, sort, logger, false);
    }

    /**
     * Internal search
     * @param searchParameters the search parameters
     * @param offset starting index
     * @param limit max number of results
     * @param sort not used
     * @param logger the logger
     * @param getRemoteValues to get remote values or not
     * @return The search result
     */
    private Map<String, Map<String, Object>> _internalSearch(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger, boolean getRemoteValues)
    {
        Map<String, Map<String, Object>> results = new LinkedHashMap<>();

        List<UserDirectory> userDirectories = new ArrayList<>();
        if (searchParameters.containsKey("userDirectory"))
        {
            userDirectories.add((UserDirectory) searchParameters.get("userDirectory"));
            searchParameters.remove("userDirectory");
        }
        else
        {
            UserPopulation population = _userPopulationDAO.getUserPopulation(getPopulationId());
            userDirectories = population.getUserDirectories();
        }

        for (UserDirectory userDirectory : userDirectories)
        {
            if (userDirectory instanceof LdapUserDirectory)
            {
                // Sort is ignored for LDAP
                results.putAll(searchLDAP((LdapUserDirectory) userDirectory, searchParameters, offset, limit, logger, getRemoteValues));
            }
            else if (userDirectory instanceof JdbcUserDirectory)
            {
                // TODO handle SQL case
                // remoteValuesByContent = searchSQL((JdbcUserDirectory) userDirectory, logger);
                logger.warn("Population with SQL datasource is not implemented.");
            }
            else
            {
                @SuppressWarnings("unchecked")
                Map<String, Map<String, Object>> userAsSearchResults = userDirectory.getStoredUsers()
                    .stream()
                    .map(this::_userToSearchResult)
                    .collect(Collectors.toMap(
                        m -> ((List<String>) m.get(getIdField())).get(0),
                        m -> m
                    ));
                
                results.putAll(userAsSearchResults);
            }
        }
        
        return results;
    }
    
    /**
     * Transform user to search result
     * @param storedUser the user
     * @return the search result
     */
    protected Map<String, Object> _userToSearchResult(StoredUser storedUser)
    {
        Map<String, Object> json = new HashMap<>();
        
        json.put(getIdField(), List.of(storedUser.getIdentifier()));
        
        String firstName = storedUser.getFirstName();
        json.put(getFirstNameAttributeName(), StringUtils.isNotBlank(firstName) ? List.of(firstName) : List.of());
        
        String lastName = storedUser.getLastName();
        json.put(getLastNameAttributeName(), StringUtils.isNotBlank(lastName) ? List.of(lastName) : List.of());
        
        String email = storedUser.getEmail();
        json.put(getEmailAttributeName(), StringUtils.isNotBlank(email) ? List.of(email) : List.of());
        
        return json;
    }
    
    @Override
    @SuppressWarnings("unchecked")
    protected Map<String, Map<String, List<Object>>> getRemoteValues(Map<String, Object> searchParameters, Logger logger)
    {
        return (Map<String, Map<String, List<Object>>>) (Object) _internalSearch(searchParameters, 0, Integer.MAX_VALUE, null, logger, true);
    }
    
    @Override
    protected Map<String, Object> getAdditionalAttributeValues(String idValue, Content content, Map<String, Object> additionalParameters, boolean create, Logger logger)
    {
        Map<String, Object> additionalRemoteValues = super.getAdditionalAttributeValues(idValue, content, additionalParameters, create, logger);
        UserIdentity user = new UserIdentity(idValue, getPopulationId());
        additionalRemoteValues.put(UserSCCConstants.USER_ATTRIBUTE_NAME, user);
        return additionalRemoteValues;
    }
    
    /**
     * Get the id of the user population
     * @return The id of user population
     */
    public String getPopulationId()
    {
        return (String) getParameterValues().get(__PARAM_POPULATION_ID);
    }
    
    @Override
    public String getIdField()
    {
        return getLoginAttributeName();
    }
    
    /**
     * Get the attribute name for the login
     * @return The the attribute name for the login
     */
    public String getLoginAttributeName()
    {
        return UserSCCConstants.USER_UNIQUE_ID_ATTRIBUTE_NAME;
    }
    
    /**
     * Get the attribute name for the first name
     * @return The the attribute name for the first name
     */
    public String getFirstNameAttributeName()
    {
        return (String) getParameterValues().get(__PARAM_FIRSTNAME_ATTRIBUTE_NAME);
    }
    
    /**
     * Get the attribute name for the last name
     * @return The the attribute name for the last name
     */
    public String getLastNameAttributeName()
    {
        return (String) getParameterValues().get(__PARAM_LASTNAME_ATTRIBUTE_NAME);
    }
    
    /**
     * Get the attribute name for the email
     * @return The the attribute name for the email
     */
    public String getEmailAttributeName()
    {
        return (String) getParameterValues().get(__PARAM_EMAIL_ATTRIBUTE_NAME);
    }
    
    /**
     * Get the additional filter for searching
     * @return The additional filter for searching
     */
    public String getAdditionalSearchFilter()
    {
        return (String) getParameterValues().get(__PARAM_ADDITIONAL_SEARCH_FILTER);
    }
    
    @Override
    public Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters)
    {
        return _syncFields;
    }
    
    private Set<String> _getUserDirectoryKeys(Map<String, Object> field, String prefix)
    {
        return field.keySet().stream()
                .filter(name -> name.startsWith(prefix))
                .collect(Collectors.toSet());
    }
    
    @Override
    protected Map<String, Object> putIdParameter(String idValue)
    {
        Map<String, Object> parameters = new HashMap<>();

        for (String userDirectory : _mapping.keySet())
        {
            List<String> remoteKeys = _mapping.get(userDirectory).get(getIdField());
            if (remoteKeys != null && remoteKeys.size() > 0)
            {
                parameters.put(userDirectory + "$" + remoteKeys.get(0), idValue);
            }
        }
        
        return parameters;
    }
    
    @Override
    protected int _deleteContents(List<Content> contentsToRemove, Logger logger)
    {
        return _deleteUserComponent.deleteContentsWithLog(contentsToRemove, Map.of(DeleteOrgUnitComponent.SCC_ID_PARAMETERS_KEY, getId()), Map.of(), logger);
    }
}
