/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory;

import java.text.Normalizer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.provider.WorkspaceSelector;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.plugins.userdirectory.page.VirtualUserDirectoryPageFactory;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;

/**
 * Component providing methods to retrieve user directory virtual pages, such as the user directory root,
 * transitional page and user page.
 */
public class UserDirectoryPageHandler extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** The avalon role. */
    public static final String ROLE = UserDirectoryPageHandler.class.getName();
    
    /** The data name for the content type of the user directory */
    public static final String CONTENT_TYPE_DATA_NAME = "user-directory-root-contenttype";
    /** The data name for the users' view to use */
    public static final String USER_VIEW_NAME = "user-directory-root-view-name";
    /** The data name for the classification attribute of the user directory */
    public static final String CLASSIFICATION_ATTRIBUTE_DATA_NAME = "user-directory-root-classification-metadata";
    /** The data name for the depth of the user directory */
    public static final String DEPTH_DATA_NAME = "user-directory-root-depth";
    /** The user directory root pages cache id */
    protected static final String ROOT_PAGES_CACHE = UserDirectoryPageHandler.class.getName() + "$rootPageIds";
    /** The user directory user pages cache id */
    protected static final String UD_PAGES_CACHE = UserDirectoryPageHandler.class.getName() + "$udPages";
    
    /** The workspace selector. */
    protected WorkspaceSelector _workspaceSelector;
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The extension point for content types */
    protected ContentTypeExtensionPoint _contentTypeEP;
    /** The cache manager */
    protected AbstractCacheManager _abstractCacheManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _workspaceSelector = (WorkspaceSelector) manager.lookup(WorkspaceSelector.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _abstractCacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        _abstractCacheManager.createMemoryCache(ROOT_PAGES_CACHE, 
                new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_CACHE_ROOT_PAGES_LABEL"),
                new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_CACHE_ROOT_PAGES_DESCRIPTION"),
                true,
                null);
        _abstractCacheManager.createMemoryCache(UD_PAGES_CACHE, 
                new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_CACHE_UD_PAGES_LABEL"),
                new I18nizableText("plugin.user-directory", "PLUGINS_USER_DIRECTORY_CACHE_UD_PAGES_DESCRIPTION"),
                true,
                null);
    }
    
    /**
     * Gets the user directory root pages from the given content type id, whatever the site.
     * @param contentTypeId The content type id
     * @return the user directory root pages.
     * @throws AmetysRepositoryException  if an error occured.
     */
    public Set<Page> getUserDirectoryRootPages(String contentTypeId) throws AmetysRepositoryException
    {
        Expression expression = new VirtualFactoryExpression(VirtualUserDirectoryPageFactory.class.getName());
        Expression contentTypeExp = new StringExpression(CONTENT_TYPE_DATA_NAME, Operator.EQ, contentTypeId);
        
        AndExpression andExp = new AndExpression(expression, contentTypeExp);
        
        String query = PageQueryHelper.getPageXPathQuery(null, null, null, andExp, null);
        
        AmetysObjectIterable<Page> pages = _resolver.query(query);
        
        return pages.stream().collect(Collectors.toSet());
    }
    
    /**
     * Gets the user directory root page of a specific content type.
     * @param siteName The site name
     * @param sitemapName The sitemap
     * @param contentTypeId The content type id
     * @return the user directory root pages.
     * @throws AmetysRepositoryException  if an error occured.
     */
    public Page getUserDirectoryRootPage(String siteName, String sitemapName, String contentTypeId) throws AmetysRepositoryException
    {
        String contentTypeIdToCompare = contentTypeId != null ? contentTypeId : "";
        
        for (Page userDirectoryRootPage : getUserDirectoryRootPages(siteName, sitemapName))
        {
            if (contentTypeIdToCompare.equals(getContentTypeId(userDirectoryRootPage)))
            {
                return userDirectoryRootPage;
            }
        }
        
        return null;
    }
    
    /**
     * Gets the user directory root pages.
     * @param siteName The site name
     * @param sitemapName The sitemap
     * @return the user directory root pages.
     * @throws AmetysRepositoryException  if an error occured.
     */
    public Set<Page> getUserDirectoryRootPages(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Set<Page> rootPages = new HashSet<>();
        
        String workspace = _workspaceSelector.getWorkspace();
        
        Cache<RootPageCacheKey, Set<String>> cache = getRootPagesCache();
        
        RootPageCacheKey key = RootPageCacheKey.of(workspace, siteName, sitemapName);
        if (cache.hasKey(key))
        {
            rootPages = cache.get(key).stream()
                    .map(this::_resolvePage)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toSet());
        }
        else
        {
            rootPages = _getUserDirectoryRootPages(siteName, sitemapName);
            Set<String> userDirectoryRootPageIds = rootPages.stream()
                    .map(Page::getId)
                    .collect(Collectors.toSet());
            cache.put(key, userDirectoryRootPageIds);
        }
        
        return rootPages;
    }
    
    private Page _resolvePage(String pageId)
    {
        try
        {
            return _resolver.resolveById(pageId);
        }
        catch (UnknownAmetysObjectException e)
        {
            // The page stored in cache may have been deleted
            return null;
        }
    }
    
    /**
     * Get the user directory root pages, without searching in the cache.
     * @param siteName the current site.
     * @param sitemapName the sitemap name.
     * @return the user directory root pages
     * @throws AmetysRepositoryException if an error occured.
     */
    protected Set<Page> _getUserDirectoryRootPages(String siteName, String sitemapName) throws AmetysRepositoryException
    {
        Expression expression = new VirtualFactoryExpression(VirtualUserDirectoryPageFactory.class.getName());
        
        String query = PageQueryHelper.getPageXPathQuery(siteName, sitemapName, null, expression, null);
        
        AmetysObjectIterable<Page> pages = _resolver.query(query);
        
        return pages.stream().collect(Collectors.toSet());
    }
    
    /**
     * Gets the depth of the user directory root page
     * @param rootPage The user directory root page
     * @return the depth of the user directory root page
     */
    public int getDepth(Page rootPage)
    {
        return Math.toIntExact(rootPage.getValue(DEPTH_DATA_NAME));
    }
    
    /**
     * Gets the name of the classification attribute
     * @param rootPage The user directory root page
     * @return the name of the classification attribute
     */
    public String getClassificationAttribute(Page rootPage)
    {
        return rootPage.getValue(CLASSIFICATION_ATTRIBUTE_DATA_NAME);
    }
    
    /**
     * Gets the content type id
     * @param rootPage The user directory root page
     * @return the content type id
     */
    public String getContentTypeId(Page rootPage)
    {
        return rootPage.getValue(CONTENT_TYPE_DATA_NAME);
    }
    
    /**
     * Gets the content type
     * @param rootPage The user directory root page
     * @return the content type
     */
    public ContentType getContentType(Page rootPage)
    {
        String contentTypeId = getContentTypeId(rootPage);
        return StringUtils.isNotBlank(contentTypeId) ? _contentTypeEP.getExtension(contentTypeId) : null;
    }
    
    /**
     * Gets the value of the classification attribute for the given content, transformed for building tree hierarchy
     * <br>The transformation takes the lower-case of all characters, removes non-alphanumeric characters,
     * and takes the first characters to not have a string with a size bigger than the depth
     * <br>For instance, if the value for the content is "Aéa Foo-bar" and the depth is 7,
     * then this method will return "aeafoob"
     * @param rootPage The user directory root page
     * @param content The content
     * @return the transformed value of the classification attribute for the given content. Can be null
     */
    public String getTransformedClassificationValue(Page rootPage, Content content)
    {
        String attribute = getClassificationAttribute(rootPage);
        int depth = getDepth(rootPage);
        
        // 1) get value of the classification attribute
        String classification = content.getValue(attribute);
        
        if (classification == null)
        {
            // The classification does not exists for the content
            getLogger().info("The classification attribute '{}' does not exist for the content {}", attribute, content);
            return null;
        }
        
        try
        {
            // 2) replace special character
            // 3) remove '-' characters
            
            // FIXME CMS-5758 FilterNameHelper.filterName do not authorized name with numbers only.
            // So code of FilterNamehelper is temporarily duplicated here with a slightly modified RegExp
//            String transformedValue = FilterNameHelper.filterName(classification).replace("-", "");
            String transformedValue = _filterName(classification).replace("-", "");
            
            // 4) only keep 'depth' first characters (if depth = 3, "de" becomes "de", "debu" becomes "deb", etc.)
            return StringUtils.substring(transformedValue, 0, depth);
        }
        catch (IllegalArgumentException e)
        {
            // The value of the classification attribute is not valid
            getLogger().warn("The classification attribute '{}' does not have a valid value ({}) for the content {}", attribute, classification, content);
            return null;
        }
    }
    
    private String _filterName(String name)
    {
        Pattern pattern = Pattern.compile("^()[0-9-_]*[a-z0-9].*$");
        // Use lower case
        // then remove accents
        // then replace contiguous spaces with one dash
        // and finally remove non-alphanumeric characters except -
        String filteredName = Normalizer.normalize(name.toLowerCase(), Normalizer.Form.NFD).replaceAll("[\\p{InCombiningDiacriticalMarks}]", "").trim(); 
        filteredName = filteredName.replaceAll("œ", "oe").replaceAll("æ", "ae").replaceAll(" +", "-").replaceAll("[^\\w-]", "-").replaceAll("-+", "-");

        Matcher m = pattern.matcher(filteredName);
        if (!m.matches())
        {
            throw new IllegalArgumentException(filteredName + " doesn't match the expected regular expression : " + pattern.pattern());
        }

        filteredName = filteredName.substring(m.end(1));

        // Remove characters '-' and '_' at the start and the end of the string
        return StringUtils.strip(filteredName, "-_");
    }
    
    /**
     * Get all transitional page child from page name
     * @param rootPage the root page
     * @param pagePath the page path
     * @return all transitional page child from page name
     */
    public SortedSet<String> getTransitionalPagesName(Page rootPage, String pagePath)
    {
        String workspace = _workspaceSelector.getWorkspace();
        String site = rootPage.getSiteName();
        String contentType = getContentTypeId(rootPage);
        String lang = rootPage.getSitemapName();
        
        PageCacheKey key = PageCacheKey.of(workspace, contentType, site, lang);
        UDPagesCache udCache = getUDPagesCache().get(key, k -> _getUDPages(rootPage, workspace, contentType, lang));

        Map<String, SortedSet<String>> transitionalPages = udCache.transitionalPagesCache();
        String cachePagePath = getName(pagePath);
        return transitionalPages.getOrDefault(cachePagePath, new TreeSet<>());
    }
    
    /**
     * Get all user page child from page name
     * @param rootPage the root page
     * @param pagePath the page path
     * @return all user page child from page name
     */
    public Map<String, String> getUserPagesContent(Page rootPage, String pagePath)
    {
        String workspace = _workspaceSelector.getWorkspace();
        String site = rootPage.getSiteName();
        String contentType = getContentTypeId(rootPage);
        String lang = rootPage.getSitemapName();
        
        PageCacheKey key = PageCacheKey.of(workspace, contentType, site, lang);
        UDPagesCache udCache = getUDPagesCache().get(key, k -> _getUDPages(rootPage, workspace, contentType, lang));
        
        Map<String, Map<String, String>> userPages = udCache.userPagesCache();
        String cachePagePath = getName(pagePath);
        return userPages.getOrDefault(cachePagePath, new HashMap<>());
    }
    
    /**
     * Get the UD cache by page path
     * For transitional pages returning a map as {'p' : [a, e], 'p/a' : [], 'p/e' : []} 
     * For user pages returning a map as {'p' : {userContent1: user://xxxxxxx1, userContent2: user://xxxxxxx2,}, 'p/a' : {userContent1: user://xxxxxxx1}, 'p/e' : {userContent2: user://xxxxxxx2}} 
     * @param rootPage the root page
     * @param workspace the workspace
     * @param contentType the content type
     * @param lang the language
     * @return the UD pages cache
     */
    private UDPagesCache _getUDPages(Page rootPage, String workspace, String contentType, String lang)
    {
        // Getting all user content with its classification identifier defined in the root page
        Map<Content, String> transformedValuesByContent = _getTransformedValuesByContent(rootPage);
        
        // Computing transitional pages cache
        Set<String> transformedValues = new HashSet<>(transformedValuesByContent.values());
        Map<String, SortedSet<String>> transitionalPagesCache = _getTransitionalPageByPagePath(transformedValues);

        // Computing user pages cache
        int depth = getDepth(rootPage);
        Map<String, Map<String, String>> userPageCache = _getUserContentsByPagePath(transformedValuesByContent, depth);
        
        getLogger().info("UD pages cache was built for workspace '{}' and content type '{}' and language '{}'", workspace, contentType, lang);
        return new UDPagesCache(transitionalPagesCache, userPageCache);
    }
    
    private Map<String, SortedSet<String>> _getTransitionalPageByPagePath(Set<String> transformedValues)
    {
        Map<String, SortedSet<String>> transitionalPageByPath = new HashMap<>();
        for (String value : transformedValues)
        {
            char[] charArray = value.toCharArray();
            for (int i = 0; i < charArray.length; i++)
            {
                String lastChar = String.valueOf(charArray[i]);
                if (i == 0)
                {
                    // case _root
                    SortedSet<String> root = transitionalPageByPath.getOrDefault("_root", new TreeSet<>());
                    if (!root.contains(lastChar))
                    {
                        root.add(lastChar);
                    }
                    transitionalPageByPath.put("_root", root);
                }
                else
                {
                    String currentPrefixWithoutLastChar = value.substring(0, i); // if value == "debu", equals to "deb"
                    String currentPathWithoutLastChar = StringUtils.join(currentPrefixWithoutLastChar.toCharArray(), '/'); // if value == "debu", equals to "d/e/b"
                    SortedSet<String> childPageNames = transitionalPageByPath.getOrDefault(currentPathWithoutLastChar, new TreeSet<>());
                    if (!childPageNames.contains(lastChar))
                    {
                        childPageNames.add(lastChar); // if value == "debu", add "u" in childPageNames for key "d/e/b"
                    }
                    transitionalPageByPath.put(currentPathWithoutLastChar, childPageNames);
                }
            }
        }
        
        return transitionalPageByPath;
    }
    
    private Map<String, Map<String, String>> _getUserContentsByPagePath(Map<Content, String> transformedValuesByContent, int depth)
    {
        Map<String, Map<String, String>> contentsByPath = new LinkedHashMap<>();
        if (depth == 0)
        {
            Map<String, String> rootContents = new LinkedHashMap<>();
            for (Content content : transformedValuesByContent.keySet())
            {
                rootContents.put(content.getName(), content.getId());
            }
            
            contentsByPath.put("_root", rootContents);
            return contentsByPath;
        }
        
        for (Content content : transformedValuesByContent.keySet())
        {
            String transformedValue = transformedValuesByContent.get(content);
            for (int i = 0; i < depth; i++)
            {
                String currentPrefix = StringUtils.substring(transformedValue, 0, i + 1);
                String currentPath = StringUtils.join(currentPrefix.toCharArray(), '/');
                Map<String, String> contentsForPath = contentsByPath.getOrDefault(currentPath, new LinkedHashMap<>());
                
                String contentName = content.getName();
                if (!contentsForPath.containsKey(contentName))
                {
                    contentsForPath.put(contentName, content.getId());
                }
                contentsByPath.put(currentPath, contentsForPath);
            }
        }
        return contentsByPath;
    }
    
    /**
     * Get all transformed values by content
     * @param rootPage the root page 
     * @return the map of transformed values by content
     */
    protected Map<Content, String> _getTransformedValuesByContent(Page rootPage)
    {
        // Get all contents which will appear in the sitemap
        AmetysObjectIterable<Content> contents = getContentsForRootPage(rootPage);
        
        // Get their classification attribute value
        Map<Content, String> transformedValuesByContent = new LinkedHashMap<>();
        for (Content content : contents)
        {
            String value = getTransformedClassificationValue(rootPage, content);
            if (value != null)
            {
                transformedValuesByContent.put(content, value);
            }
        }
        return transformedValuesByContent;
    }

    /**
     * Get the user contents for a given root page
     * @param rootPage the root page
     * @return the user contents
     */
    public AmetysObjectIterable<Content> getContentsForRootPage(Page rootPage)
    {
        String contentType = getContentTypeId(rootPage);
        String lang = rootPage.getSitemapName();
        
        Set<String> subTypes = _contentTypeEP.getSubTypes(contentType);
        
        List<Expression> contentTypeExpressions = new ArrayList<>();
        contentTypeExpressions.add(new ContentTypeExpression(Operator.EQ, contentType));
        for (String subType : subTypes)
        {
            contentTypeExpressions.add(new ContentTypeExpression(Operator.EQ, subType));
        }
        
        Expression contentTypeExpression = new OrExpression(contentTypeExpressions.toArray(new Expression[subTypes.size() + 1]));
        
        Expression finalExpr = new AndExpression(contentTypeExpression, new LanguageExpression(Operator.EQ, lang));
        
        SortCriteria sort = new SortCriteria();
        sort.addCriterion(Content.ATTRIBUTE_TITLE, true, true);
        
        String xPathQuery = QueryHelper.getXPathQuery(null, "ametys:content", finalExpr, sort);
        
        return _resolver.query(xPathQuery);
    }
    
    /**
     * Gets name form path name
     * @param pathName the path name
     * @return the name
     */
    public String getName(String pathName)
    {
        String prefix = "page-";
        String name = "";
        for (String transitionalPageName : pathName.split("/"))
        {
            if (!name.equals(""))
            {
                name += "/";
            }
            name += StringUtils.startsWith(transitionalPageName, prefix) ? StringUtils.substringAfter(transitionalPageName, prefix) : transitionalPageName;
        }
        return name;
    }
    
    /**
     * Checks if name contains only Unicode digits and if so, prefix it with "page-"
     * @param name The page name
     * @return The potentially prefixed page name
     */
    public String getPathName(String name)
    {
        return StringUtils.isNumeric(name) ? "page-" + name : name; 
    }
    
    /**
     * Clear root page cache
     * @param rootPage the root page
     */
    public void clearCache(Page rootPage)
    {
        clearCache(getContentTypeId(rootPage));
    }
    
    /**
     * Clear root page cache
     * @param contentTypeId the content type id
     */
    public void clearCache(String contentTypeId)
    {
        getUDPagesCache().invalidate(PageCacheKey.of(null, contentTypeId, null, null));
        
        getRootPagesCache().invalidateAll();
    }
    
    /**
     * Cache of the user directory root pages.
     * The cache store a Set of TODO indexed by the workspaceName, siteName, siteMapName
     * @return the cache
     */
    protected Cache<RootPageCacheKey, Set<String>> getRootPagesCache()
    {
        return _abstractCacheManager.get(ROOT_PAGES_CACHE);
    }
    
    /**
     * Key to index a user directory root page in a cache
     */
    protected static final class RootPageCacheKey extends AbstractCacheKey
    {
        /**
         * Basic constructor
         * @param workspaceName the workspace name. Can be null.
         * @param siteName the site name. Can be null.
         * @param language the sitemap name. Can be null.
         */
        public RootPageCacheKey(String workspaceName, String siteName, String language)
        {
            super(workspaceName, siteName, language);
        }
        
        /**
         * Generate a cache key
         * @param workspaceName the workspace name. Can be null.
         * @param siteName the site name. Can be null.
         * @param language the sitemap name. Can be null.
         * @return the cache key
         */
        public static RootPageCacheKey of(String workspaceName, String siteName, String language)
        {
            return new RootPageCacheKey(workspaceName, siteName, language);
        }
    }
    
    /**
     * Cache of the user directory user pages and transitional page.
     * The cache store a {@link UDPagesCache} containing the transitional pages cache and the user pages cache.
     * The cache is indexed by workspaceName, siteName, siteMapName, pageName.
     * @return the cache
     */
    protected Cache<PageCacheKey, UDPagesCache> getUDPagesCache()
    {
        return _abstractCacheManager.get(UD_PAGES_CACHE);
    }
    
    /**
     * Key to index a user directory page in a cache
     */
    protected static final class PageCacheKey extends AbstractCacheKey
    {
        /**
         * Basic constructor
         * @param workspaceName the workspace name. Can be null.
         * @param contentTypeId the contentType id. Can be null.
         * @param siteName the site name. Can be null.
         * @param language the sitemap name. Can be null.
         */
        public PageCacheKey(String workspaceName, String contentTypeId, String siteName, String language)
        {
            super(workspaceName, contentTypeId, siteName, language);
        }
        
        /**
         * Generate a cache key
         * @param workspaceName the workspace name. Can be null.
         * @param contentTypeId the contentType id. Can be null.
         * @param siteName the site name. Can be null.
         * @param language the sitemap name. Can be null.
         * @return the cache key
         */
        public static PageCacheKey of(String workspaceName, String contentTypeId, String siteName, String language)
        {
            return new PageCacheKey(workspaceName, contentTypeId, siteName, language);
        }
    }
    
    /**
     * User directory pages cache 
     * @param transitionalPagesCache the cache for transitional pages. The cache store a {@link Map} of (content path, sorted set of transitional page path).
     * @param userPagesCache the cache for user pages. The cache store a {@link Map} of (content path, (content name, content id)) of all the content of the page.
     */
    protected record UDPagesCache(Map<String, SortedSet<String>> transitionalPagesCache, Map<String, Map<String, String>> userPagesCache) { /** */ }
}
