/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.userdirectory.clientsideelement;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.value.StringValue;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.userdirectory.OrganisationChartPageHandler;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.userdirectory.observation.ObservationConstants;
import org.ametys.plugins.userdirectory.page.VirtualOrganisationChartPageFactory;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.clientsideelement.AbstractPageClientSideElement;
import org.ametys.web.repository.page.LockablePage;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.Page;
import org.ametys.web.rights.PageRightAssignmentContext;

/**
 * Client side element for a controller wich set/remove the organisation chart root page
 */
public class SetOrganisationChartRootClientSideElement extends AbstractPageClientSideElement
{
    /** Observer manager. */
    protected ObservationManager _observationManager;

    /** The organisation chart page handler */
    protected OrganisationChartPageHandler _pageHandler;
    
    /** The extension point for content types */
    protected ContentTypeExtensionPoint _contentTypeEP;
    
    /** The organization chart page handler */
    protected OrganisationChartPageHandler _orgUnitPageHandler;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
        _pageHandler = (OrganisationChartPageHandler) smanager.lookup(OrganisationChartPageHandler.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _orgUnitPageHandler = (OrganisationChartPageHandler) smanager.lookup(OrganisationChartPageHandler.ROLE);
    }
    /**
     * Gets the status of the given page
     * @param pageId The page id
     * @return the status of the given page
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getStatus(String pageId)
    {
        // Assume that no read access is checked (required to update client side element status)
        
        Map<String, Object> result = new HashMap<>();
        
        Map<String, Object> parameters = this._script.getParameters();
        
        Page page = _resolver.resolveById(pageId);

        if (page instanceof JCRAmetysObject)
        {
            if (_pageHandler.isOrganisationChartRootPage((JCRAmetysObject) page))
            {
                List<String> i18nParameters = new ArrayList<>();
                i18nParameters.add(page.getTitle());
    
                I18nizableText ed = (I18nizableText) parameters.get("organisation-chart-page-description");
                I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                result.put("organisation-chart-page-title", msg);
                
                ed = (I18nizableText) parameters.get("remove-organisation-chart-page-description");
                msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                result.put("remove-organisation-chart-page-title", msg);
                
                String contentTypeId = page.getValue(OrganisationChartPageHandler.CONTENT_TYPE_DATA_NAME, StringUtils.EMPTY);
                
                if (StringUtils.isNotEmpty(contentTypeId))
                {
                    I18nizableText contentTypeText = _contentTypeEP.hasExtension(contentTypeId) ? _contentTypeEP.getExtension(contentTypeId).getLabel() : new I18nizableText(contentTypeId);
                    
                    Map<String, I18nizableTextParameter> contentTypeI18nParameters = new HashMap<>();
                    contentTypeI18nParameters.put("0", contentTypeText);
                    
                    ed = (I18nizableText) parameters.get("contenttype-organisation-chart-page-description");
                    msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), contentTypeI18nParameters);
                    result.put("contenttype-organisation-chart-page-description", msg);
                }
                
                result.put("organisation-chart-page-id", new I18nizableText(page.getId()));
            }
            else
            {
                List<String> i18nParameters = new ArrayList<>();
                i18nParameters.add(page.getTitle());
    
                I18nizableText ed = (I18nizableText) parameters.get("add-organisation-chart-page-description");
                I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
                
                result.put("add-organisation-chart-page-id", new I18nizableText(page.getId()));
                result.put("add-organisation-chart-page-title", msg);
            }
        }
        else
        {
            List<String> noJcrI18nParameters = new ArrayList<>();
            noJcrI18nParameters.add(page.getTitle());

            I18nizableText ed = (I18nizableText) parameters.get("no-jcr-page-description");
            I18nizableText msg = new I18nizableText(ed.getCatalogue(), ed.getKey(), noJcrI18nParameters);
            
            result.put("no-jcr-page-id", new I18nizableText(page.getId()));
            result.put("no-jcr-page-title", msg);
        }
        
        return result;
    }
    
    /**
     * Sets the given page as the root of a organization chart
     * @param pageId The id of the page
     * @param contentType The id of the content type
     * @param pageVisible true to make the organization page visible
     * @return A result map
     * @throws RepositoryException if a repository error occurred
     */
    @Callable (rights = "User_Directory_Right_Organisation_Chart_SetRoot", rightContext = PageRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> setOrganisationChartRoot(String pageId, String contentType, boolean pageVisible) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        if (!_contentTypeEP.isSameOrDescendant(contentType, UserDirectoryHelper.ORGUNIT_CONTENT_TYPE))
        {
            result.put("error", "invalid-content-type");
            return result;
        }
        
        Page page = _resolver.resolveById(pageId);
        
        if (page instanceof LockablePage lockablePage && lockablePage.isLocked())
        {
            throw new IllegalStateException("Cannot set the locked page '/" + page.getSitemapName() + "/" + page.getPathInSitemap() + "' as the root of a organization chart");
        }

        String oldContentType = page.getValue(OrganisationChartPageHandler.CONTENT_TYPE_DATA_NAME, StringUtils.EMPTY);
        boolean oldVisiblity = page.getValue(OrganisationChartPageHandler.PAGE_VISIBLE_DATA_NAME, true);
        
        // Do nothing if page attribute are the same
        if (!oldContentType.equals(contentType) || oldVisiblity != pageVisible)
        {
            Set<Page> currentOrgUnitPages = _orgUnitPageHandler.getOrganisationChartRootPages(page.getSiteName(), page.getSitemapName());
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
            
            if (currentOrgUnitPages.contains(page))
            {
                // Unindex pages for all workspaces before the properties changed
                _observationManager.notify(new Event(ObservationConstants.EVENT_ORGANISATION_CHART_ROOT_UPDATING, _currentUserProvider.getUser(), eventParams));
                
                _updateOrgUnitRootProperty(page, contentType, pageVisible);
            }
            else
            {
                _addOrgUnitRootProperty(page, contentType, pageVisible);
            }
            
            
            // Live synchronization
            _notifyPageUpdated(page);
            
            // Indexation
            _observationManager.notify(new Event(ObservationConstants.EVENT_ORGANISATION_CHART_ROOT_UPDATED, _currentUserProvider.getUser(), eventParams));
        }
        
        return result;
    }
    private void _addOrgUnitRootProperty(Page page, String contentType, boolean pageVisible) throws RepositoryException
    {
        if (page instanceof JCRAmetysObject)
        {
            JCRAmetysObject jcrPage = (JCRAmetysObject) page;
            Node node = jcrPage.getNode();
            
            List<Value> values = new ArrayList<>();
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                values.addAll(Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues()));
            }
            
            StringValue virtualOrgUnitPageFactoryClassName = new StringValue(VirtualOrganisationChartPageFactory.class.getName());
            if (!values.contains(virtualOrgUnitPageFactoryClassName))
            {
                values.add(virtualOrgUnitPageFactoryClassName);
            }
            
            node.setProperty(AmetysObjectResolver.VIRTUAL_PROPERTY, values.toArray(new Value[values.size()]));

            // Set the organisation chart root property
            if (page instanceof ModifiablePage)
            {
                ModifiablePage modifiablePage = (ModifiablePage) page;
                modifiablePage.setValue(OrganisationChartPageHandler.CONTENT_TYPE_DATA_NAME, contentType);
                modifiablePage.setValue(OrganisationChartPageHandler.PAGE_VISIBLE_DATA_NAME, pageVisible);
            }
            
            jcrPage.saveChanges();
        }
    }
    
    private void _updateOrgUnitRootProperty(Page page, String contentType, boolean pageVisible)
    {
        if (page instanceof ModifiablePage)
        {
            ModifiablePage modifiablePage = (ModifiablePage) page;
            
            // Set the organisation chart root property
            modifiablePage.setValue(OrganisationChartPageHandler.CONTENT_TYPE_DATA_NAME, contentType);
            modifiablePage.setValue(OrganisationChartPageHandler.PAGE_VISIBLE_DATA_NAME, pageVisible);
            
            modifiablePage.saveChanges();
        }
    }
    /**
     * Remove the organization chart root status to the given page
     * @param pageId The id of the page
     * @return A result map
     * @throws RepositoryException if a repository error occurred
     */
    @Callable (rights = "User_Directory_Right_Organisation_Chart_SetRoot", rightContext = PageRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> removeOrganisationChartRoot(String pageId) throws RepositoryException
    {
        Map<String, Object> result = new HashMap<>();
        
        Page page = _resolver.resolveById(pageId);
        
        if (page instanceof JCRAmetysObject)
        {
            if (!_pageHandler.isOrganisationChartRootPage((JCRAmetysObject) page))
            {
                result.put("error", "no-root");
                return result;
            }
            
            if (page instanceof LockablePage lockablePage && lockablePage.isLocked())
            {
                throw new IllegalStateException("Cannot unset root status from a locked page '/" + page.getSitemapName() + "/" + page.getPathInSitemap() + "'");
            }
            
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
            
            // Unindex pages for all workspaces before the properties were removed
            _observationManager.notify(new Event(ObservationConstants.EVENT_ORGANISATION_CHART_ROOT_DELETING, _currentUserProvider.getUser(), eventParams));
            
            _removeOrgUnitRootProperty(page);
            
            _notifyPageUpdated(page);
            
            // After live synchronization
            _observationManager.notify(new Event(ObservationConstants.EVENT_ORGANISATION_CHART_ROOT_DELETED, _currentUserProvider.getUser(), eventParams));
        }
        else
        {
            result.put("error", "no-root");
        }
        return result;
    }

    
    /**
     * Gets the content types which can build an organisation chart
     * @param pageId The id of the page being edited
     * @return the content types which can build an organisation chart
     */
    @Callable (rights = "User_Directory_Right_Organisation_Chart_SetRoot", rightContext = PageRightAssignmentContext.ID, paramIndex = 0)
    public List<Map<String, Object>> getSupportedContentTypes(String pageId)
    {
        List<Map<String, Object>> result = new ArrayList<>();
        Page page = _resolver.resolveById(pageId);
        
        Set<String> orgUnitContentTypes = new HashSet<>();
                
        orgUnitContentTypes.add(UserDirectoryHelper.ORGUNIT_CONTENT_TYPE);
        orgUnitContentTypes.addAll(_contentTypeEP.getSubTypes(UserDirectoryHelper.ORGUNIT_CONTENT_TYPE));
        
        for (String contentTypeId : orgUnitContentTypes)
        {
            ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
            Page orgUnitRootPage = _orgUnitPageHandler.getOrganisationChartRootPage(page.getSiteName(), page.getSitemapName(), contentTypeId);
            if (!contentType.isAbstract() && (orgUnitRootPage == null || orgUnitRootPage.equals(page)))
            {
                // The content type is not already a root of an organisation chart or is the root of the currently edited page
                Map<String, Object> entry = new HashMap<>();
                entry.put("value", contentType.getId());
                entry.put("text", contentType.getLabel());
                result.add(entry);
            }
        }
        
        return result;
    }
    
    private void _removeOrgUnitRootProperty(Page page) throws RepositoryException
    {
        if (page instanceof JCRAmetysObject)
        {
            JCRAmetysObject jcrPage = (JCRAmetysObject) page;
            Node node = jcrPage.getNode();
            
            if (node.hasProperty(AmetysObjectResolver.VIRTUAL_PROPERTY))
            {
                List<Value> values = new ArrayList<>(Arrays.asList(node.getProperty(AmetysObjectResolver.VIRTUAL_PROPERTY).getValues()));
                int index = values.stream()
                        .map(LambdaUtils.wrap(Value::getString))
                        .collect(Collectors.toList())
                        .indexOf(VirtualOrganisationChartPageFactory.class.getName());
                
                if (index != -1)
                {
                    values.remove(index);
                    node.setProperty(AmetysObjectResolver.VIRTUAL_PROPERTY, values.toArray(new Value[values.size()]));
                }

                // Remove the organization chart root property
                if (page instanceof ModifiablePage)
                {
                    ModifiablePage modifiablePage = (ModifiablePage) page;
                    modifiablePage.removeValue(OrganisationChartPageHandler.CONTENT_TYPE_DATA_NAME);
                    modifiablePage.removeValue(OrganisationChartPageHandler.PAGE_VISIBLE_DATA_NAME);
                }
                jcrPage.saveChanges();
            }
        }
    }
    
    private void _notifyPageUpdated(Page page)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
        _observationManager.notify(new Event(org.ametys.web.ObservationConstants.EVENT_PAGE_UPDATED, _currentUserProvider.getUser(), eventParams));
    }

    
    /**
     * Gets information about organisation chart root status on the given.
     * @param pageId The id of the page
     * @return information about organisation chart root status on the given.
     */
    @Callable (rights = "User_Directory_Right_Organisation_Chart_SetRoot", rightContext = PageRightAssignmentContext.ID, paramIndex = 0)
    public Map<String, Object> getRootPageInfo(String pageId)
    {
        Map<String, Object> result = new HashMap<>();
        
        Page page = _resolver.resolveById(pageId);
        
        if (page.hasValue(OrganisationChartPageHandler.CONTENT_TYPE_DATA_NAME))
        {
            result.put("isRoot", true);
            result.put("contentType", page.getValue(OrganisationChartPageHandler.CONTENT_TYPE_DATA_NAME));
            result.put("pageVisible", page.getValue(OrganisationChartPageHandler.PAGE_VISIBLE_DATA_NAME, true));
        }
        else
        {
            result.put("isRoot", false);
        }
        
        return result;
    }
}
